﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/


using UnityEngine;
using UnityEditor;
using System.Collections.Generic;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Manages Editor callbacks and events.
	/// </summary>
	[InitializeOnLoad]
	public static class HEU_EditorApp
	{

		/// <summary>
		/// Executed after script (re)load. Sets up plugin callbacks.
		/// </summary>
		static HEU_EditorApp()
		{
			EditorApplication.hierarchyWindowItemOnGUI += HierarchyWindowItemOnGUI;

			SceneView.onSceneGUIDelegate += OnSceneGUIDelegate;
		}

		private static void HierarchyWindowItemOnGUI(int instanceID, Rect selectionRect)
		{
			ProcessDragEvent(Event.current, null);
		}

		private static void OnSceneGUIDelegate(SceneView sceneView)
		{
			ProcessDragEvent(Event.current, sceneView);
		}

		private static void ProcessDragEvent(Event dragEvent, SceneView sceneView)
		{
			if(dragEvent != null && (dragEvent.type == EventType.DragUpdated || dragEvent.type == EventType.DragPerform))
			{
				bool dragHDAs = false;
				List<string> hdaList = new List<string>();
				foreach(string file in DragAndDrop.paths)
				{
					if(HEU_HAPIUtility.IsHoudiniAssetFile(file))
					{
						dragHDAs = true;
						DragAndDrop.visualMode = DragAndDropVisualMode.Move;
						hdaList.Add(file);
						break;
					}
				}

				if(dragHDAs)
				{
					if (dragEvent.type == EventType.DragPerform)
					{
						if (HEU_SessionManager.ValidatePluginSession())
						{
							Vector3 dropPos = Vector3.zero;
							if (sceneView != null)
							{
								Camera camera = sceneView.camera;
								Vector3 mousePos = HEU_EditorUI.GetMousePosition(ref dragEvent, camera);

								Ray ray = camera.ScreenPointToRay(mousePos);
								ray.origin = camera.transform.position;
								Plane plane = new Plane();
								plane.SetNormalAndPosition(Vector3.up, Vector3.zero);
								float enter = 0f;
								plane.Raycast(ray, out enter);
								enter = Mathf.Clamp(enter, camera.nearClipPlane, camera.farClipPlane);
								dropPos = ray.origin + ray.direction * enter;
							}

							List<GameObject> createdGOs = new List<GameObject>();
							foreach (string file in hdaList)
							{
								GameObject go = HEU_HAPIUtility.InstantiateHDA(file, dropPos, HEU_SessionManager.GetOrCreateDefaultSession());
								if(go != null)
								{
									createdGOs.Add(go);
								}
							}

							// Select the created assets
							HEU_EditorUtility.SelectObjects(createdGOs.ToArray());
						}
					}

					dragEvent.Use();
				}
			}
		}
	}

}