﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using UnityEditor;

using System.IO;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Unity Editor menu functions
	/// </summary>
	public class HEU_EditorMenu : MonoBehaviour
	{
		// SESSIONS ---------------------------------------------------------------------------------------------------

#if (UNITY_EDITOR_64 || UNITY_64)
		/* Commenting out In-Process sessions as its not recommended usage
		// In-Process session is only available in 64-bit Houdini Engine library
		[MenuItem(HEU_Constants.HEU_PRODUCT_NAME + "/Session/Create/" + HEU_EditorStrings.INPROCESS_SESSION, false, 0)]
		public static void CreateInProcessSession()
		{
			bool bResult = HEU_SessionManager.CreateInProcessSession();
			if (!bResult)
			{
				HEU_EditorUtility.DisplayErrorDialog("Create Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}
		*/
#endif

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/Create/" + HEU_EditorStrings.RPC_PIPE_SESSION, false, 0)]
		public static void CreatePipeSession()
		{
			bool bResult = HEU_SessionManager.CreateThriftPipeSession(HEU_PluginSettings.Session_PipeName, HEU_PluginSettings.Session_AutoClose, HEU_PluginSettings.Session_Timeout);
			if (!bResult)
			{
				HEU_EditorUtility.DisplayErrorDialog("Create Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/Create/" + HEU_EditorStrings.RPC_SOCKET_SESSION, false, 0)]
		public static void CreateSocketSession()
		{
			bool bResult = HEU_SessionManager.CreateThriftSocketSession(HEU_PluginSettings.Session_Localhost, HEU_PluginSettings.Session_Port, HEU_PluginSettings.Session_AutoClose, HEU_PluginSettings.Session_Timeout);
			if (!bResult)
			{
				HEU_EditorUtility.DisplayErrorDialog("Create Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/Connect To Debugger/" + HEU_EditorStrings.RPC_PIPE_SESSION, false, 0)]
		public static void DebugConnectPipeSession()
		{
			bool bResult = HEU_SessionManager.ConnectThriftPipeSession(HEU_PluginSettings.Session_PipeName, HEU_PluginSettings.Session_AutoClose, HEU_PluginSettings.Session_Timeout);
			if (!bResult)
			{
				HEU_EditorUtility.DisplayErrorDialog("Debug Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/Connect To Debugger/" + HEU_EditorStrings.RPC_SOCKET_SESSION, false, 0)]
		public static void DebugConnectSocketSession()
		{
			bool bResult = HEU_SessionManager.ConnectThriftSocketSession(HEU_PluginSettings.Session_Localhost, HEU_PluginSettings.Session_Port, HEU_PluginSettings.Session_AutoClose, HEU_PluginSettings.Session_Port);
			if (!bResult)
			{
				HEU_EditorUtility.DisplayErrorDialog("Debug Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/" + HEU_EditorStrings.GET_SESSION_INFO, false, 20)]
		public static void GetSessionInfo()
		{
			HEU_EditorUtility.DisplayDialog("Houdini Engine", HEU_SessionManager.GetSessionInfo(), "OK");
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/" + HEU_EditorStrings.CLOSE_DEFAULT_SESSION, false, 40)]
		public static void CloseDefaultSession()
		{
			bool bResult = HEU_SessionManager.CloseDefaultSession();
			if (!bResult)
			{
				HEU_EditorUtility.DisplayErrorDialog("Closing Default Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
			else
			{
				Debug.Log("Houdini Engine Session closed!");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/" + HEU_EditorStrings.CLOSE_ALL_SESSIONS, false, 40)]
		public static void CloseAllSessions()
		{
			HEU_SessionManager.CloseAllSessions();
			Debug.Log("Houdini Engine Sessions closed!");
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/" + HEU_EditorStrings.RECONNECT_TO_SESSION, false, 60)]
		public static void ReconnectToSession()
		{
			bool bResult = HEU_SessionManager.LoadStoredDefaultSession();
			if (!bResult)
			{
				HEU_EditorUtility.DisplayDialog("Reconnecting to Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
			else
			{
				Debug.Log("Houdini Engine Session reconnected.");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Session/" + HEU_EditorStrings.RESTART_SESSION, false, 60)]
		public static void ReinitializeSession()
		{
			bool bResult = HEU_SessionManager.RestartSession();
			if(!bResult)
			{
				HEU_EditorUtility.DisplayDialog("Reinitializing Session", HEU_SessionManager.GetLastSessionError(), "OK");
			}
			else
			{
				Debug.Log("Houdini Engine Session restarted.");
			}
		}


		// INSTALLATION -----------------------------------------------------------------------------------------------

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/" + "Installation Info", false, 0)]
		public static void ShowInstallationInfo()
		{
			HEU_EditorUtility.DisplayDialog(HEU_Defines.HEU_INSTALL_INFO, HEU_HAPIUtility.GetHoudiniEngineInstallationInfo(), "OK");
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/" + "Plugin Settings", false, 0)]
		public static void ShowSettingsWindow()
		{
			HEU_SettingsWindow.ShowWindow();
		}

		// DEBUG ---------------------------------------------------------------------------------------------------

		/* COMMENTED OUT FOR NOW UNTIL FULL IMPLEMENTATION IS IN
		[MenuItem(HEU_Constants.HEU_PRODUCT_NAME + "/Debug/" + HEU_EditorStrings.LOAD_SESSION_FROM_HIP, false, 40)]
		public static void LoadSessionFromHIP()
		{
			bool bResult = HEU_SessionManager.LoadSessionFromHIP(true);
			if (!bResult)
			{
				HEU_EditorUtility.DisplayDialog("Loading Session From HIP", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}
		*/

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Debug/" + HEU_EditorStrings.OPEN_SCENE_IN_HOUDINI, false, 20)]
		public static void OpenSceneHoudini()
		{
			bool bResult = HEU_SessionManager.OpenSessionInHoudini();
			if (!bResult)
			{
				HEU_EditorUtility.DisplayDialog("Opening Session in Houdini", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/Debug/" + HEU_EditorStrings.SAVE_SESSION_TO_HIP, false, 20)]
		public static void SaveSessionToHIP()
		{
			bool bResult = HEU_SessionManager.SaveSessionToHIP(false);
			if (!bResult)
			{
				HEU_EditorUtility.DisplayDialog("Saving Session to HIP", HEU_SessionManager.GetLastSessionError(), "OK");
			}
		}

		// GENERATE ---------------------------------------------------------------------------------------------------

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/" + "Load Houdini Digital Asset", false, 40)]
		public static void LoadHoudiniAssetWindow()
		{
			if (HEU_SessionManager.ValidatePluginSession())
			{
				string extesions = "otl;*.hda;*.otllc;*.hdalc;*.otlnc;*.hdanc";
				string hdaPath = EditorUtility.OpenFilePanel("Load Houdini Digital Asset", HEU_PluginSettings.LastLoadHDAPath, extesions);
				if (!string.IsNullOrEmpty(hdaPath))
				{
					// Store HDA path for next time
					HEU_PluginSettings.LastLoadHDAPath = Path.GetDirectoryName(hdaPath);

					GameObject go = HEU_HAPIUtility.InstantiateHDA(hdaPath, Vector3.zero, HEU_SessionManager.GetOrCreateDefaultSession());
					if (go != null)
					{
						HEU_EditorUtility.SelectObject(go);
					}
				}
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/" + "New Curve Asset", false, 60)]
		public static void CreateNewCurveAsset()
		{
			GameObject newCurveGO = HEU_HAPIUtility.CreateNewCurveAsset();
			if(newCurveGO != null)
			{
				HEU_Curve.PreferredNextInteractionMode = HEU_Curve.Interaction.ADD;
				HEU_EditorUtility.SelectObject(newCurveGO);
			}
		}

		[MenuItem(HEU_Defines.HEU_PRODUCT_NAME + "/" + "New Input Asset", false, 60)]
		public static void CreateNewInputAsset()
		{
			GameObject newCurveGO = HEU_HAPIUtility.CreateNewInputAsset();
			if (newCurveGO != null)
			{
				HEU_EditorUtility.SelectObject(newCurveGO);
			}
		}
	}


}   // HoudiniEngineUnity