﻿/*
* Copyright (c) <2018> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using UnityEditor;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Editor drawing logic for input nodes.
	/// </summary>
	public static class HEU_InputNodeUI
	{
		/// <summary>
		/// Populate the UI cache for the given input node
		/// </summary>
		/// <param name="inputNode"></param>
		public static void PopulateCache(HEU_InputNode inputNode)
		{
			if (inputNode._uiCache == null)
			{
#if UNITY_EDITOR
				inputNode._uiCache = new HEU_InputNodeUICache();

				inputNode._uiCache._inputNodeSerializedObject = new SerializedObject(inputNode);

				inputNode._uiCache._inputObjectTypeProperty = HEU_EditorUtility.GetSerializedProperty(inputNode._uiCache._inputNodeSerializedObject, "_inputObjectType");
				inputNode._uiCache._keepWorldTransformProperty = HEU_EditorUtility.GetSerializedProperty(inputNode._uiCache._inputNodeSerializedObject, "_keepWorldTransform");
				inputNode._uiCache._packBeforeMergeProperty = HEU_EditorUtility.GetSerializedProperty(inputNode._uiCache._inputNodeSerializedObject, "_packGeometryBeforeMerging");

				inputNode._uiCache._inputObjectsProperty = HEU_EditorUtility.GetSerializedProperty(inputNode._uiCache._inputNodeSerializedObject, "_inputObjects");

				inputNode._uiCache._inputAssetProperty = HEU_EditorUtility.GetSerializedProperty(inputNode._uiCache._inputNodeSerializedObject, "_inputAsset");

				int inputCount = inputNode._uiCache._inputObjectsProperty.arraySize;
				for (int i = 0; i < inputCount; ++i)
				{
					SerializedProperty inputObjectProperty = inputNode._uiCache._inputObjectsProperty.GetArrayElementAtIndex(i);

					HEU_InputNodeUICache.HEU_InputObjectUICache objectCache = new HEU_InputNodeUICache.HEU_InputObjectUICache();

					objectCache._gameObjectProperty = inputObjectProperty.FindPropertyRelative("_gameObject");

					objectCache._transformOverrideProperty = inputObjectProperty.FindPropertyRelative("_useTransformOverride");

					objectCache._translateProperty = inputObjectProperty.FindPropertyRelative("_translateOverride");
					objectCache._rotateProperty = inputObjectProperty.FindPropertyRelative("_rotateOverride");
					objectCache._scaleProperty = inputObjectProperty.FindPropertyRelative("_scaleOverride");

					inputNode._uiCache._inputObjectCache.Add(objectCache);
				}
#endif
			}
		}

		/// <summary>
		/// Draw the UI for the given input node
		/// </summary>
		/// <param name="inputNode"></param>
		public static void EditorDrawInputNode(HEU_InputNode inputNode)
		{
			int plusButtonWidth = 20;

			//GUIStyle boldLabelStyle = new GUIStyle(EditorStyles.boldLabel);
			//boldLabelStyle.alignment = TextAnchor.UpperLeft;

			GUIContent translateLabel = new GUIContent("    Translate");
			GUIContent rotateLabel = new GUIContent("    Rotate");
			GUIContent scaleLabel = new GUIContent("    Scale");

			PopulateCache(inputNode);

			EditorGUI.BeginChangeCheck();

			EditorGUILayout.BeginVertical(EditorStyles.helpBox);

			string inputName = inputNode.InputName;
			if (!string.IsNullOrEmpty(inputName))
			{
				EditorGUILayout.LabelField(inputName);
			}

			EditorGUI.indentLevel++;

			HEU_InputNode.InputObjectType inputObjectType = (HEU_InputNode.InputObjectType)inputNode._uiCache._inputObjectTypeProperty.enumValueIndex;
			HEU_InputNode.InputObjectType userSelectedInputObjectType = (HEU_InputNode.InputObjectType)EditorGUILayout.EnumPopup(inputObjectType);
			if (userSelectedInputObjectType != inputObjectType)
			{
				SerializedProperty pendingInputObjectTypeProperty = HEU_EditorUtility.GetSerializedProperty(inputNode._uiCache._inputNodeSerializedObject, "_pendingInputObjectType");
				if(pendingInputObjectTypeProperty != null)
				{
					pendingInputObjectTypeProperty.enumValueIndex = (int)userSelectedInputObjectType;
				}
			}
			else
			{
				if (inputObjectType == HEU_InputNode.InputObjectType.HDA)
				{
					EditorGUILayout.PropertyField(inputNode._uiCache._inputAssetProperty);
				}
				//else if (inputObjectType == HEU_InputNode.InputObjectType.CURVE)
				//{
				//	TODO INPUT CURVE
				//}
				else if (inputObjectType == HEU_InputNode.InputObjectType.UNITY_MESH)
				{
					EditorGUILayout.PropertyField(inputNode._uiCache._keepWorldTransformProperty);
					EditorGUILayout.PropertyField(inputNode._uiCache._packBeforeMergeProperty);

					SerializedProperty inputObjectsProperty = inputNode._uiCache._inputObjectsProperty;
					if (inputObjectsProperty != null)
					{
						int inputCount = inputObjectsProperty.arraySize;
						bool bSkipElements = false;

						HEU_EditorUI.DrawSeparator();

						using (var hs1 = new EditorGUILayout.HorizontalScope())
						{
							EditorGUILayout.LabelField(string.Format("{0} input objects", inputCount));

							if (GUILayout.Button("Add"))
							{
								inputObjectsProperty.InsertArrayElementAtIndex(inputCount);
								bSkipElements = true;
							}

							if (GUILayout.Button("Clear"))
							{
								inputObjectsProperty.ClearArray();
								bSkipElements = true;
							}
						}

						if (!bSkipElements)
						{
							using (var vs1 = new EditorGUILayout.VerticalScope())
							{
								for (int i = 0; i < inputCount; ++i)
								{
									using (var hs2 = new EditorGUILayout.HorizontalScope())
									{
										EditorGUILayout.LabelField("Input " + (i + 1));

										//using (var vs3 = new EditorGUILayout.VerticalScope())
										{
											if (GUILayout.Button("+", GUILayout.Width(plusButtonWidth)))
											{
												inputObjectsProperty.InsertArrayElementAtIndex(i);
												break;
											}

											if (GUILayout.Button("-", GUILayout.Width(plusButtonWidth)))
											{
												inputObjectsProperty.DeleteArrayElementAtIndex(i);
												break;
											}
										}
									}

									EditorGUI.indentLevel++;
									using (var vs4 = new EditorGUILayout.VerticalScope())
									{
										HEU_InputNodeUICache.HEU_InputObjectUICache objectCache = inputNode._uiCache._inputObjectCache[i];

										EditorGUILayout.PropertyField(objectCache._gameObjectProperty, GUIContent.none);

										objectCache._transformOverrideProperty.boolValue = HEU_EditorUI.DrawToggleLeft(objectCache._transformOverrideProperty.boolValue, "Transform Override");
										if (objectCache._transformOverrideProperty.boolValue)
										{
											objectCache._translateProperty.vector3Value = EditorGUILayout.Vector3Field(translateLabel, objectCache._translateProperty.vector3Value);
											objectCache._rotateProperty.vector3Value = EditorGUILayout.Vector3Field(rotateLabel, objectCache._rotateProperty.vector3Value);
											objectCache._scaleProperty.vector3Value = EditorGUILayout.Vector3Field(scaleLabel, objectCache._scaleProperty.vector3Value);
										}
									}
									EditorGUI.indentLevel--;
								}
							}
						}
					}
				}
			}

			EditorGUI.indentLevel--;

			EditorGUILayout.EndVertical();

			if (EditorGUI.EndChangeCheck())
			{
				inputNode._uiCache._inputNodeSerializedObject.ApplyModifiedProperties();

				// When cooking, this will force input data to be uploaded
				inputNode.RequiresUpload = true;
			}
		}
	}


}	// HoudiniEngineUnity