﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using System.Collections.Generic;
using UnityEngine;
using UnityEditor;

namespace HoudiniEngineUnity
{

	/// <summary>
	/// Custom editor to draw HDA parameters.
	/// This will handle all the drawing for all parameters in an HDA.
	/// Requires an HEU_Parameters that has been properly setup with parameter hierarchy.
	/// </summary>
	[CustomEditor(typeof(HEU_Parameters))]
	public class HEU_ParameterUI : Editor
	{
		// Cache reference to the currently selected HDA's parameter object
		private HEU_Parameters _parameters;

		// Cached list of parameter objects property
		private SerializedProperty _rootParametersProperty;
		private SerializedProperty _parameterListProperty;

		private SerializedProperty _showParametersProperty;

		// Layout constants
		private const float _multiparmPlusMinusWidth = 40f;


		// Store cached SerializedProperties of a HEU_ParameterData class, and store as list
		private class HEU_ParameterUICache
		{
			public HEU_ParameterData _parameterData;

			public SerializedProperty _paramType;
			public SerializedProperty _primaryValue;
			public SerializedProperty _secondaryValue;

			// For multiparm instance, this would be its index into the multiparm list
			public int _instanceIndex = -1;
			// For multiparm instance, this is the index of the parameter for this instance
			public int _paramIndexForInstance = -1;

			public List<HEU_ParameterUICache> _childrenCache;

			public string[] _tabLabels;
		}
		private List<HEU_ParameterUICache> _parameterCache;

		private static GUIStyle _sliderStyle;
		private static GUIStyle _sliderThumbStyle;




		private void OnEnable()
		{
			_parameters = target as HEU_Parameters;
			//Debug.Assert(_parameters != null, "Target is not HEU_Parameters");
		}

		private void OnDisable()
		{
			_rootParametersProperty = null;
			_parameterListProperty = null;
			_showParametersProperty = null;

			_parameterCache = null;
		}

		public override void OnInspectorGUI()
		{
			// First check if serialized properties need to be cached
			if (_parameterCache == null || _parameters.RecacheUI)
			{
				CacheProperties();
			}

			InitializeGUIStyles();

			float previousFieldWidth = EditorGUIUtility.fieldWidth;
			EditorGUIUtility.fieldWidth = 100;

			// Now draw the cached properties

			EditorGUI.BeginChangeCheck();

			HEU_EditorUI.BeginSection();

			// Draw all the parameters. Start at root parameters, and draw their children recursively.
			_showParametersProperty.boolValue = HEU_EditorUI.DrawFoldOut(_showParametersProperty.boolValue, _parameters._uiLabel);
			if (_showParametersProperty.boolValue)
			{
				if (!_parameters.AreParametersValid())
				{
					EditorGUILayout.LabelField("Parameters haven't been generated or failed to generate.\nPlease Recook or Rebuild to regenerate them!", GUILayout.MinHeight(35));
				}
				else if (_parameters.RequiresRegeneration || _parameters.HasModifiersPending())
				{
					// Skip drawing parameters if there pending changes. 
					// Cooking needs to happen before we let the user manipulate UI further.
					EditorGUILayout.LabelField("Parameters have changed.\nPlease Recook to upload them to Houdini!", GUILayout.MinHeight(35));
				}
				else
				{
					foreach (HEU_ParameterUICache paramCache in _parameterCache)
					{
						DrawParamUICache(paramCache);
					}
				}
			}

			HEU_EditorUI.EndSection();

			if (EditorGUI.EndChangeCheck())
			{
				serializedObject.ApplyModifiedProperties();
			}

			EditorGUIUtility.fieldWidth = previousFieldWidth;
		}

		private static void InitializeGUIStyles()
		{
			if (_sliderStyle == null)
			{
				_sliderStyle = new GUIStyle(GUI.skin.horizontalSlider);
			}
			if (_sliderThumbStyle == null)
			{
				_sliderThumbStyle = new GUIStyle(GUI.skin.horizontalSliderThumb);
			}
		}

		private void CacheProperties()
		{
			//Debug.Log("Cacheing Properties");
			serializedObject.Update();

			// Flag that we've cached
			_parameters.RecacheUI = false;

			_rootParametersProperty = HEU_EditorUtility.GetSerializedProperty(serializedObject, "_rootParameters");
			_parameterListProperty = HEU_EditorUtility.GetSerializedProperty(serializedObject, "_parameterList");

			_showParametersProperty = HEU_EditorUtility.GetSerializedProperty(serializedObject, "_showParameters");

			// Cache each parameter based on its type
			_parameterCache = new List<HEU_ParameterUICache>();
			int paramCount = _rootParametersProperty.arraySize;
			for (int i = 0; i < paramCount; ++i)
			{
				// Get each root level index
				SerializedProperty elementProperty = _rootParametersProperty.GetArrayElementAtIndex(i);
				int childListIndex = elementProperty.intValue;

				// Find the parameter data associated with the index
				SerializedProperty childParameterProperty = _parameterListProperty.GetArrayElementAtIndex(childListIndex);

				HEU_ParameterUICache newParamUI = ProcessParamUICache(childListIndex, childParameterProperty);
				if(newParamUI != null)
				{
					_parameterCache.Add(newParamUI);
				}
			}
		}

		private HEU_ParameterUICache ProcessParamUICache(int parameterIndex, SerializedProperty parameterProperty)
		{
			// Get the actual parameter data associated with this parameter
			HEU_ParameterData parameterData = _parameters.GetParameter(parameterIndex);

			HEU_ParameterUICache newParamUICache = null;
			if (parameterData.IsMultiParam())
			{
				newParamUICache = ProcessMultiParamUICache(parameterData, parameterProperty);
			}
			else if (parameterData.IsContainer())
			{
				newParamUICache = ProcessContainerParamUICache(parameterData, parameterProperty);
			}
			else
			{
				newParamUICache = ProcessLeafParameterCache(parameterData, parameterProperty);
			}

			return newParamUICache;
		}

		private HEU_ParameterUICache ProcessMultiParamUICache(HEU_ParameterData parameterData, SerializedProperty parameterProperty)
		{
			HEU_ParameterUICache paramUICache = null;

			if (parameterData._parmInfo.rampType == HAPI_RampType.HAPI_RAMPTYPE_COLOR)
			{
				paramUICache = new HEU_ParameterUICache();
				paramUICache._parameterData = parameterData;
				paramUICache._childrenCache = new List<HEU_ParameterUICache>();

				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_childParameterIDs");
				Debug.Assert(paramUICache._primaryValue != null && paramUICache._primaryValue.arraySize > 0, "Multiparams should have at least 1 child");

				paramUICache._secondaryValue = parameterProperty.FindPropertyRelative("_gradient");

				// For ramps, the number of instances is the number of points in the ramp
				// Each point can then have a number of parameters.
				int numPoints = parameterData._parmInfo.instanceCount;
				int numParamsPerPoint = parameterData._parmInfo.instanceLength;
				int childIndex = 0;

				for (int pointIndex = 0; pointIndex < numPoints; ++pointIndex)
				{
					for (int paramIndex = 0; paramIndex < numParamsPerPoint; ++paramIndex)
					{
						SerializedProperty childIndexProperty = paramUICache._primaryValue.GetArrayElementAtIndex(childIndex);
						SerializedProperty childParameterProperty = _parameterListProperty.GetArrayElementAtIndex(childIndexProperty.intValue);

						HEU_ParameterUICache newChildParamUI = ProcessParamUICache(childIndexProperty.intValue, childParameterProperty);
						if(newChildParamUI != null)
						{
							paramUICache._childrenCache.Add(newChildParamUI);

							newChildParamUI._paramIndexForInstance = newChildParamUI._parameterData._parmInfo.childIndex;
							
							// Using instance index to store the int param index
							newChildParamUI._instanceIndex = newChildParamUI._parameterData._parmInfo.instanceNum;
						}

						childIndex++;
					}
				}
			}
			else if (parameterData._parmInfo.rampType == HAPI_RampType.HAPI_RAMPTYPE_FLOAT)
			{
				paramUICache = new HEU_ParameterUICache();
				paramUICache._parameterData = parameterData;
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_childParameterIDs");
			}
			else
			{
				paramUICache = new HEU_ParameterUICache();
				paramUICache._parameterData = parameterData;

				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_childParameterIDs");

				if (paramUICache._primaryValue != null)
				{
					paramUICache._childrenCache = new List<HEU_ParameterUICache>();

					for (int i = 0; i < paramUICache._primaryValue.arraySize; ++i)
					{
						SerializedProperty childIndexProperty = paramUICache._primaryValue.GetArrayElementAtIndex(i);
						int childIndex = childIndexProperty.intValue;
						SerializedProperty childParameterProperty = _parameterListProperty.GetArrayElementAtIndex(childIndex);
						if (childParameterProperty != null)
						{
							HEU_ParameterUICache newChildParamUI = ProcessParamUICache(childIndex, childParameterProperty);

							if (newChildParamUI != null)
							{
								paramUICache._childrenCache.Add(newChildParamUI);

								newChildParamUI._instanceIndex = newChildParamUI._parameterData._parmInfo.instanceNum;
								newChildParamUI._paramIndexForInstance = newChildParamUI._parameterData._parmInfo.childIndex;
							}
						}
					}
				}
			}

			return paramUICache;
		}

		private HEU_ParameterUICache ProcessContainerParamUICache(HEU_ParameterData parameterData, SerializedProperty parameterProperty)
		{
			HEU_ParameterUICache paramUICache = new HEU_ParameterUICache();
			paramUICache._parameterData = parameterData;

			//Debug.LogFormat("Container: name={0}, type={1}, size={2}, chidlren={3}", parameterData._name, parameterData._parmInfo.type, parameterData._parmInfo.size, parameterData._childParameterIDs.Count);

			paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_childParameterIDs");
			if (paramUICache._primaryValue != null && paramUICache._primaryValue.arraySize > 0)
			{
				paramUICache._secondaryValue = parameterProperty.FindPropertyRelative("_showChildren");
				paramUICache._childrenCache = new List<HEU_ParameterUICache>();

				// Process child parameters
				for (int i = 0; i < paramUICache._primaryValue.arraySize; ++i)
				{
					SerializedProperty childIndexProperty = paramUICache._primaryValue.GetArrayElementAtIndex(i);
					int childIndex = childIndexProperty.intValue;
					SerializedProperty childParameterProperty = _parameterListProperty.GetArrayElementAtIndex(childIndex);
					if (childParameterProperty != null)
					{
						HEU_ParameterUICache newChildUICache = ProcessParamUICache(childIndex, childParameterProperty);
						if(newChildUICache != null)
						{
							paramUICache._childrenCache.Add(newChildUICache);
						}
					}
				}

				// Fill in children (folder) labels for folder list
				if (paramUICache._parameterData._parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_FOLDERLIST)
				{
					paramUICache._tabLabels = new string[paramUICache._childrenCache.Count];
					for (int i = 0; i < paramUICache._childrenCache.Count; ++i)
					{
						paramUICache._tabLabels[i] = paramUICache._childrenCache[i]._parameterData._labelName;
					}

					paramUICache._secondaryValue = parameterProperty.FindPropertyRelative("_tabSelectedIndex");
				}
			}

			return paramUICache;
		}

		private HEU_ParameterUICache ProcessLeafParameterCache(HEU_ParameterData parameterData, SerializedProperty parameterProperty)
		{
			HEU_ParameterUICache paramUICache = new HEU_ParameterUICache();
			paramUICache._parameterData = parameterData;

			paramUICache._paramType = parameterProperty.FindPropertyRelative("_parmInfo.type");
			HAPI_ParmType parmType = (HAPI_ParmType)paramUICache._paramType.intValue;

			if (parmType == HAPI_ParmType.HAPI_PARMTYPE_INT)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_intValues");

				if (parameterData._parmInfo.choiceCount > 0)
				{
					paramUICache._secondaryValue = parameterProperty.FindPropertyRelative("_choiceValue");
				}
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_FLOAT)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_floatValues");
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_STRING)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_stringValues");

				if (parameterData._parmInfo.choiceCount > 0)
				{
					paramUICache._secondaryValue = parameterProperty.FindPropertyRelative("_choiceValue");
				}
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_TOGGLE)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_toggle");
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_COLOR)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_color");
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_BUTTON)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_intValues");
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_PATH_FILE || parmType == HAPI_ParmType.HAPI_PARMTYPE_PATH_FILE_GEO
				|| parmType == HAPI_ParmType.HAPI_PARMTYPE_PATH_FILE_IMAGE)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_stringValues");
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_NODE)
			{
				paramUICache._primaryValue = parameterProperty.FindPropertyRelative("_paramInputNode");
			}
			else if(parmType == HAPI_ParmType.HAPI_PARMTYPE_SEPARATOR || parmType == HAPI_ParmType.HAPI_PARMTYPE_LABEL)
			{
				// Allow these
			}
			else
			{
				// Ignore all others (null out the cache)
				paramUICache = null;
			}

			return paramUICache;
		}

		private void DrawParamUICache(HEU_ParameterUICache paramCache, bool bDrawFoldout = true)
		{
			if (paramCache._parameterData.IsMultiParam())
			{
				DrawMultiParamUICache(paramCache);
			}
			else if (paramCache._parameterData.IsContainer())
			{
				DrawContainerParamUICache(paramCache, bDrawFoldout);
			}
			else
			{
				DrawLeafParamUICache(paramCache);
			}
		}

		/// <summary>
		/// Draws array properties in the Inspector UI with proper rows and columns.
		/// Uses delayed draws to reduce cooking on each character change in a field.
		/// </summary>
		/// <param name="labelString"></param>
		/// <param name="arrayProperty"></param>
		private void DrawArrayProperty(string labelString, SerializedProperty arrayProperty)
		{
			// Arrays are drawn with a label, and rows of values.

			GUILayout.BeginHorizontal();
			{
				EditorGUILayout.PrefixLabel(labelString);

				GUILayout.BeginVertical(EditorStyles.helpBox);
				{
					int numElements = arrayProperty.arraySize;
					int maxElementsPerRow = 4;

					GUILayout.BeginHorizontal();
					{
						for (int i = 0; i < numElements; ++i)
						{
							if (i > 0 && i % maxElementsPerRow == 0)
							{
								GUILayout.EndHorizontal();
								GUILayout.BeginHorizontal();
							}

							switch (arrayProperty.GetArrayElementAtIndex(0).propertyType)
							{
								case SerializedPropertyType.Integer:
								{
									EditorGUILayout.DelayedIntField(arrayProperty.GetArrayElementAtIndex(i), GUIContent.none);
									break;
								}
								case SerializedPropertyType.Float:
								{
									EditorGUILayout.DelayedFloatField(arrayProperty.GetArrayElementAtIndex(i), GUIContent.none);
									break;
								}
								case SerializedPropertyType.String:
								{
									EditorGUILayout.DelayedTextField(arrayProperty.GetArrayElementAtIndex(i), GUIContent.none);
									break;
								}
							}
						}
					}
					GUILayout.EndHorizontal();
				}
				GUILayout.EndVertical();
			}
			GUILayout.EndHorizontal();
		}

		private void DrawContainerParamUICache(HEU_ParameterUICache paramUICache, bool bDrawFoldout = true)
		{
			HEU_ParameterData parameterData = paramUICache._parameterData;

			if (paramUICache._childrenCache != null)
			{
				if (paramUICache._parameterData._parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_FOLDERLIST && paramUICache._childrenCache.Count > 1)
				{
					// For folder list with multiple children, draw as tabs

					EditorGUILayout.BeginVertical(EditorStyles.helpBox);
					{
						int tabChoice = GUILayout.Toolbar(paramUICache._secondaryValue.intValue, paramUICache._tabLabels);
						if (tabChoice >= 0 && tabChoice < paramUICache._childrenCache.Count)
						{
							paramUICache._secondaryValue.intValue = tabChoice;
							DrawParamUICache(paramUICache._childrenCache[paramUICache._secondaryValue.intValue], false);
						}
					}
					EditorGUILayout.EndVertical();
				}
				else
				{
					// Otherwise draw as foldout, unless requested not to (latter case when drawing folder under a folder list)

					// If folder list, then only have 1 child, so don't draw as foldout. Also due to using paramUICache._secondaryValue
					// as tab index rather than foldout show/hide state.
					if (paramUICache._parameterData._parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_FOLDERLIST || string.IsNullOrEmpty(parameterData._labelName))
					{
						bDrawFoldout = false;
					}

					if (bDrawFoldout)
					{
						EditorGUILayout.BeginVertical(EditorStyles.helpBox);
						paramUICache._secondaryValue.boolValue = EditorGUILayout.Foldout(paramUICache._secondaryValue.boolValue, parameterData._labelName, EditorStyles.foldout);
					}

					if (!bDrawFoldout || paramUICache._secondaryValue.boolValue)
					{
						foreach (HEU_ParameterUICache paramCache in paramUICache._childrenCache)
						{
							DrawParamUICache(paramCache);
						}
					}

					if (bDrawFoldout)
					{
						EditorGUILayout.EndVertical();
					}
				}
			}
		}

		private void DrawLeafParamUICache(HEU_ParameterUICache paramUICache)
		{
			// This is a leaf parameter. So draw based on type.

			HEU_ParameterData parameterData = paramUICache._parameterData;

			HAPI_ParmType parmType = (HAPI_ParmType)paramUICache._paramType.intValue;
			if (parmType == HAPI_ParmType.HAPI_PARMTYPE_INT)
			{
				SerializedProperty intsProperty = paramUICache._primaryValue;

				if (parameterData._parmInfo.choiceCount > 0)
				{
					// Drop-down choice list for INTs

					// Get the current choice value
					SerializedProperty choiceProperty = paramUICache._secondaryValue;

					// Draw it as an int popup, passing in the user options (_choiceLabels), and the corresponding Houdini values (_choiceIntValues)
					EditorGUILayout.IntPopup(choiceProperty, parameterData._choiceLabels, parameterData._choiceIntValues, new GUIContent(parameterData._labelName));

					// No need to check, just always updated with latest choiceProperty
					if (intsProperty.GetArrayElementAtIndex(0).intValue != parameterData._choiceIntValues[choiceProperty.intValue])
					{
						//Debug.LogFormat("Setting int property {0} from {1} to {2}", parameterData._labelName, intsProperty.GetArrayElementAtIndex(0).intValue, parameterData._choiceIntValues[choiceProperty.intValue]);
						intsProperty.GetArrayElementAtIndex(0).intValue = parameterData._choiceIntValues[choiceProperty.intValue];
					}
				}
				else
				{
					if (intsProperty.arraySize == 1)
					{
						bool bHasUIMinMax = (parameterData.HasUIMin() && parameterData.HasUIMax());

						int value = intsProperty.GetArrayElementAtIndex(0).intValue;

						EditorGUILayout.BeginHorizontal();
						{
							value = EditorGUILayout.DelayedIntField(new GUIContent(parameterData._labelName), value, GUILayout.ExpandWidth(!bHasUIMinMax));
							if (bHasUIMinMax)
							{
								value = Mathf.RoundToInt(GUILayout.HorizontalSlider(value, parameterData.IntUIMin, parameterData.IntUIMax, _sliderStyle, _sliderThumbStyle));
							}
						}
						EditorGUILayout.EndHorizontal();

						if (parameterData.HasMin())
						{
							value = Mathf.Max(parameterData.IntMin, value);
						}
						if (parameterData.HasMax())
						{
							value = Mathf.Min(parameterData.IntMax, value);
						}
						intsProperty.GetArrayElementAtIndex(0).intValue = value;
					}
					else
					{
						// Multiple ints. Display label, then each element.

						DrawArrayProperty(parameterData._labelName, intsProperty);
					}
				}
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_FLOAT)
			{
				SerializedProperty floatsProperty = paramUICache._primaryValue;

				if (floatsProperty.arraySize == 1)
				{
					// Draw single float as either a slider (if min & max are available, or simply as a field)

					bool bHasUIMinMax = (parameterData.HasUIMin() && parameterData.HasUIMax());

					float value = floatsProperty.GetArrayElementAtIndex(0).floatValue;

					EditorGUILayout.BeginHorizontal();
					{
						value = EditorGUILayout.DelayedFloatField(new GUIContent(parameterData._labelName), value, GUILayout.ExpandWidth(!bHasUIMinMax));
						if (bHasUIMinMax)
						{
							value = GUILayout.HorizontalSlider(value, parameterData.FloatUIMin, parameterData.FloatUIMax, _sliderStyle, _sliderThumbStyle);
						}
					}
					EditorGUILayout.EndHorizontal();

					if (parameterData.HasMin())
					{
						value = Mathf.Max(parameterData.FloatMin, value);
					}
					if (parameterData.HasMax())
					{
						value = Mathf.Min(parameterData.FloatMax, value);
					}
					floatsProperty.GetArrayElementAtIndex(0).floatValue = value;
				}
				else
				{
					// Multiple floats. Display label, then each element.

					DrawArrayProperty(parameterData._labelName, floatsProperty);
				}

			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_STRING)
			{
				SerializedProperty stringsProperty = paramUICache._primaryValue;

				if (parameterData._parmInfo.choiceCount > 0)
				{
					// Dropdown choice list for STRINGS

					// Get the current choice value
					SerializedProperty choiceProperty = paramUICache._secondaryValue;

					// Draw it as an int popup, passing in the user options (_choiceLabels), and the corresponding Houdini values (_choiceIntValues)
					EditorGUILayout.IntPopup(choiceProperty, parameterData._choiceLabels, parameterData._choiceIntValues, new GUIContent(parameterData._labelName));

					// choiceProperty.intValue now holds the user's choice, so just update it
					stringsProperty.GetArrayElementAtIndex(0).stringValue = parameterData._choiceStringValues[choiceProperty.intValue];
				}
				else
				{
					// Draw strings as list or singularly
					DrawArrayProperty(parameterData._labelName, stringsProperty);
				}
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_TOGGLE)
			{
				EditorGUILayout.PropertyField(paramUICache._primaryValue, new GUIContent(parameterData._labelName));
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_COLOR)
			{
				EditorGUILayout.PropertyField(paramUICache._primaryValue, new GUIContent(parameterData._labelName));
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_BUTTON)
			{
				SerializedProperty intsProperty = paramUICache._primaryValue;
				Debug.Assert(intsProperty.arraySize == 1, "Button parameter property should have only a single value!");

				if (GUILayout.Button(parameterData._labelName))
				{
					intsProperty.GetArrayElementAtIndex(0).intValue = intsProperty.GetArrayElementAtIndex(0).intValue == 0 ? 1 : 0;
				}
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_PATH_FILE || parmType == HAPI_ParmType.HAPI_PARMTYPE_PATH_FILE_GEO
				|| parmType == HAPI_ParmType.HAPI_PARMTYPE_PATH_FILE_IMAGE)
			{
				GUIStyle boxStyle = HEU_EditorUI.GetGUIStyle("Groupbox", 4, 0);
				using (var vs = new EditorGUILayout.VerticalScope(boxStyle))
				{
					GUIContent labelContent = new GUIContent(parameterData._labelName);
					EditorGUILayout.LabelField(labelContent);

					EditorGUILayout.BeginHorizontal();

					SerializedProperty stringsProperty = paramUICache._primaryValue;
					Debug.Assert(stringsProperty.arraySize == 1, "File path parameter property should only have a single value!");
					EditorGUILayout.DelayedTextField(stringsProperty.GetArrayElementAtIndex(0), GUIContent.none);

					GUIStyle buttonStyle = HEU_EditorUI.GetNewButtonStyle_MarginPadding(0, 0);
					if (GUILayout.Button("...", buttonStyle, GUILayout.Width(30), GUILayout.Height(18)))
					{
						string filePattern = parameterData._fileTypeInfo;
						if (string.IsNullOrEmpty(filePattern))
						{
							filePattern = "*";
						}
						else
						{
							filePattern.Replace(" ", ";");
							if (filePattern.StartsWith("*."))
							{
								filePattern = filePattern.Substring(2);
							}
							else if (filePattern.StartsWith("*"))
							{
								filePattern = filePattern.Substring(1);
							}
						}

						string userFilePath = null;
						if (parameterData._parmInfo.permissions == HAPI_Permissions.HAPI_PERMISSIONS_WRITE_ONLY)
						{
							userFilePath = EditorUtility.SaveFilePanel("Select File", stringsProperty.GetArrayElementAtIndex(0).stringValue, "", filePattern);
							
						}
						else
						{
							userFilePath = EditorUtility.OpenFilePanel("Select File", stringsProperty.GetArrayElementAtIndex(0).stringValue, filePattern);
						}
						if (!string.IsNullOrEmpty(userFilePath))
						{
							stringsProperty.GetArrayElementAtIndex(0).stringValue = HEU_Platform.GetValidRelativePath(userFilePath);
						}
					}

					EditorGUILayout.EndHorizontal();
				}
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_SEPARATOR)
			{
				// Drawing twice give Unity's version more vertical space (similar to Houdini separator size)
				HEU_EditorUI.DrawSeparator();
				HEU_EditorUI.DrawSeparator();
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_LABEL)
			{
				HEU_EditorUI.DrawHeadingLabel(parameterData._labelName);
			}
			else if (parmType == HAPI_ParmType.HAPI_PARMTYPE_NODE)
			{
				HEU_InputNode inputNode = paramUICache._primaryValue.objectReferenceValue as HEU_InputNode;
				if (inputNode != null)
				{
					HEU_InputNodeUI.EditorDrawInputNode(inputNode);
				}
			}
			else
			{
				//Debug.LogFormat("Param name={0}, type={1}, folderType={2}", parameterData._labelName, parameterData._parmInfo.type, parameterData._parmInfo.fo);
			}
		}

		private void DrawMultiParamUICache(HEU_ParameterUICache paramUICache)
		{
			// Multiparams are drawn in its own section. Not doing foldout for it.

			HEU_ParameterData parameterData = paramUICache._parameterData;

			if (parameterData._parmInfo.rampType == HAPI_RampType.HAPI_RAMPTYPE_COLOR)
			{
				DrawColorRampParamUICache(paramUICache);
			}
			else if (parameterData._parmInfo.rampType == HAPI_RampType.HAPI_RAMPTYPE_FLOAT)
			{
				DrawFloatRampParamUICache(paramUICache);
			}
			else
			{
				// Non-ramp multiparams

				int numInstances = parameterData._parmInfo.instanceCount;
				int oldNumInstances = numInstances;

				int instanceStartOfset = parameterData._parmInfo.instanceStartOffset;

				GUILayout.BeginVertical(EditorStyles.helpBox);
				{
					// Show the top level information for a MultiParam:
					///	MultiParam Label [number of instances]  +  -  Clear
					GUILayout.BeginHorizontal();
					{
						EditorGUILayout.PrefixLabel(parameterData._labelName);
						numInstances = EditorGUILayout.DelayedIntField(numInstances);

						if (GUILayout.Button("+"))
						{
							numInstances++;
						}
						else if (numInstances > 0 && GUILayout.Button("-"))
						{
							numInstances--;
						}
						else if (numInstances > 0 && GUILayout.Button("Clear"))
						{
							numInstances = 0;
						}
					}
					GUILayout.EndHorizontal();

					// If number of instances have changed, add the modifier to handle it after UI is drawn
					if (numInstances == 0 && oldNumInstances > 0)
					{
						// Notify to clear all instances
						_parameters.ClearInstancesFromMultiParm(parameterData._unityIndex);
					}
					else if (oldNumInstances > numInstances)
					{
						// Notify to remove last set of instances
						int removeNum = (oldNumInstances - numInstances);
						_parameters.RemoveInstancesFromMultiParm(parameterData._unityIndex, instanceStartOfset + numInstances, removeNum);
					}
					else if (oldNumInstances < numInstances)
					{
						// Notify to insert new instances at end
						_parameters.InsertInstanceToMultiParm(parameterData._unityIndex, instanceStartOfset + oldNumInstances, (numInstances - oldNumInstances));
					}
					else if (numInstances == parameterData._parmInfo.instanceCount)
					{
						// As long as number of instances haven't changed, we can draw them

						for (int i = 0; i < paramUICache._childrenCache.Count; ++i)
						{
							HEU_ParameterUICache childUICache = paramUICache._childrenCache[i];
							GUILayout.BeginHorizontal();
							{
								/* TODO: commenting ability to insert/remove multiparm instances until bug #85329 has been fixed
								if(childUICache._paramIndexForInstance == 0)
								{
									if (GUILayout.Button("x", GUILayout.Width(_multiparmPlusMinusWidth)))
									{
										// Notify to remove this instance
										// Cant use _parmInfo.instanceNum here as its not the instance index. It doesn't reindex after removing instances.
										// So use the number of instances drawn so far.
										_parameters.RemoveInstancesFromMultiParm(parameterData._unityIndex, childUICache._instanceIndex, 1);

										break;
									}
									else if (GUILayout.Button("+", GUILayout.Width(_multiparmPlusMinusWidth)))
									{
										// Notify to insert new instance before this one
										_parameters.InsertInstanceToMultiParm(parameterData._unityIndex, childUICache._instanceIndex, 1);

										break;
									}
								}
								*/
							}
							DrawParamUICache(childUICache);

							GUILayout.EndHorizontal();
						}
					}
				}
				GUILayout.EndVertical();
			}
		}

		private void DrawColorRampParamUICache(HEU_ParameterUICache paramUICache)
		{
			// For a color ramp:
			//	multiparm # instances is the points on ramp (don't show that, but use as # of points)
			//	x , + is on the left side, with ramp color on right
			//	Point No.	: current point ID
			//	Position	: current point's position
			//	Color		: color value
			//	Interpolation: current point interpolation

			SerializedProperty childParameterIDsProperty = paramUICache._primaryValue;
			Debug.Assert(childParameterIDsProperty != null && childParameterIDsProperty.arraySize > 0, "Multiparams should have at least 1 child");

			HEU_ParameterData parameterData = paramUICache._parameterData;

			// For ramps, the number of instances is the number of points in the ramp
			// Each point can then have a number of parameters.

			GUILayout.BeginVertical(EditorStyles.helpBox);
			{
				EditorGUILayout.PrefixLabel(parameterData._labelName + " - (Color Ramp not supported yet)", EditorStyles.helpBox);

#if RAMP_NOT_SUPPORTED
				// Draw the gradient
				SerializedProperty gradientProperty = paramUICache._secondaryValue;
				EditorGUILayout.PropertyField(gradientProperty, GUIContent.none, true);

				// TODO: possibly use this reflection to get changes in the gradient? https://gist.github.com/capnslipp/8516384

				for(int childIndex = 0; childIndex < paramUICache._childrenCache.Count; ++childIndex)
				{
					HEU_ParameterUICache childUICache = paramUICache._childrenCache[childIndex];

					if (childUICache._paramIndexForInstance == 0)
					{
						GUILayout.BeginHorizontal(EditorStyles.helpBox);
						{
							// Draw the add and remove for each point
							if (GUILayout.Button("x"))
							{
								// TODO: remove this point
							}
							else if (GUILayout.Button("+"))
							{
								// TODO: add another point after this
							}
						}
					}

					GUILayout.BeginVertical();
					{
						if (childUICache._paramIndexForInstance == 0)
						{
							HEU_ParameterData firstPointParameter = _parameters.GetParameter(childUICache._instanceIndex);
							EditorGUILayout.PrefixLabel("Point No. " + firstPointParameter._parmInfo.instanceNum);
						}

						DrawParamUICache(childUICache);
					}
					GUILayout.EndVertical();


					if (childUICache._paramIndexForInstance == 0)
					{
						GUILayout.EndHorizontal();
					}
				}
#endif
			}
			GUILayout.EndVertical();
		}

		private void DrawFloatRampParamUICache(HEU_ParameterUICache paramUICache)
		{
			// For a float ramp:
			//	multiparm # instances is the points on ramp (don't show that, but thats # of points)
			//	x , + is on the left side, with ramp color on right
			//	Point No.	: current point ID (with slider)
			//	Position	: current point's position (with slider)
			//	Value		: float value (with slider 0 to 1)
			//	Interpolation: current point interpolation

			//Debug.Log("TODO: Draw Float Ramp");

			//SerializedProperty childParameterIDsProperty = paramUICache._primaryValue;
			HEU_ParameterData parameterData = paramUICache._parameterData;

			GUILayout.BeginVertical(EditorStyles.helpBox);
			{
				EditorGUILayout.PrefixLabel(parameterData._labelName + " - (Float Ramp not supported yet)", EditorStyles.helpBox);
			}
			GUILayout.EndVertical();
		}
	}

}   // HoudiniEngineUnity