﻿/*
* Copyright (c) <2018> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using UnityEditor;
using System.Collections.Generic;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Editor UI class for asset custom tools like Paint and Edit.
	/// </summary>
	[CanEditMultipleObjects]
	[CustomEditor(typeof(HEU_AttributesStore))]
	public class HEU_ToolsUI : Editor
	{
		// CONSTANTS --------------------------------------------------------------------------------------------------

		private const string _toolsLabel = "HOUDINI ENGINE TOOLS";

		private const string _paintValuesLabel = "Paint Values";

		private GUIContent[] _interactionModeLabels = new GUIContent[]
		{
			new GUIContent(ToolInteractionMode.VIEW.ToString()),
			new GUIContent(ToolInteractionMode.PAINT.ToString()),
			new GUIContent(ToolInteractionMode.EDIT.ToString())
		};

		private const float _mouseWheelBrushSizeMultiplier = 0.05f;
		private const float _intersectionRayLength = 5000f;

		// Keys
		private const KeyCode _brushResizeKey = KeyCode.LeftShift;


		// CACHE ------------------------------------------------------------------------------------------------------

		public List<HEU_AttributesStore> _attributesStores = new List<HEU_AttributesStore>();

		private Dictionary<HEU_AttributesStore, SerializedObject> _serializedAttributesStoresCache = new Dictionary<HEU_AttributesStore, SerializedObject>();

		private Camera _currentCamera;

		public enum ToolInteractionMode
		{
			VIEW,
			PAINT,
			EDIT
		}

		private ToolInteractionMode _interactionMode;

		private Rect _editorUIRect;

		private SerializedObject _toolsInfoSerializedObject;

		private HEU_AttributesStore _selectedAttributesStore;

		private HEU_AttributeData _selectedAttributeData;

		private SerializedProperty _selectedToolsValuesProperty;

		private bool _GUIChanged;


		// LOGIC ------------------------------------------------------------------------------------------------------


		private void OnEnable()
		{
			CacheAttributesStores();

			// Callback will be used to disable this tool and reset state
			Selection.selectionChanged += SelectionChangedCallback;
		}

		/// <summary>
		/// Callback when selection has changed.
		/// </summary>
		private void SelectionChangedCallback()
		{
			Selection.selectionChanged -= SelectionChangedCallback;

			DisableUI();
		}

		private void DisableUI()
		{
			if(_selectedAttributesStore != null)
			{
				_selectedAttributesStore.DisableEditMode();
				_selectedAttributesStore = null;
			}

			_selectedAttributeData = null;

			_attributesStores.Clear();
			_serializedAttributesStoresCache.Clear();
		}

		public void CacheAttributesStores()
		{
			//Debug.Log("Cacheing Attributes Stores!!");
			_attributesStores.Clear();
			foreach (Object targetObject in targets)
			{
				HEU_AttributesStore attributeStore = targetObject as HEU_AttributesStore;
				if (attributeStore != null)
				{
					_attributesStores.Add(attributeStore);
				}
			}
		}

		private SerializedObject GetOrCreateSerializedAttributesStore(HEU_AttributesStore attributesStore)
		{
			SerializedObject serializedAttributesStore = null;
			if (!_serializedAttributesStoresCache.TryGetValue(attributesStore, out serializedAttributesStore))
			{
				serializedAttributesStore = new SerializedObject(attributesStore);
			}
			return serializedAttributesStore;
		}

		public void DrawToolsInspectorUI(HEU_HoudiniAsset asset, SerializedObject assetObject)
		{
			if (_attributesStores.Count == 0)
			{
				return;
			}

			SerializedProperty toolsEditorProperty = HEU_EditorUtility.GetSerializedProperty(assetObject, "_toolsEditorEnabled");
			if (toolsEditorProperty != null)
			{
				EditorGUILayout.PropertyField(toolsEditorProperty);
			}

			foreach (HEU_AttributesStore attributeStore in _attributesStores)
			{
				// Debug display attributes
				EditorGUILayout.LabelField("Attribute for geo: " + attributeStore.GeoID);
			}

			// Show Tool Scene UI (TODO)
		}

		private void DrawSceneInfoPanel(HEU_HoudiniAsset asset)
		{
			float pixelsPerPoint = HEU_EditorUI.GetPixelsPerPoint();
			float screenWidth = Screen.width / pixelsPerPoint;
			float screenHeight = Screen.height / pixelsPerPoint;

			GUIStyle textBackground = new GUIStyle(GUI.skin.GetStyle("WindowBackground"));

			_toolsInfoSerializedObject = new SerializedObject(asset.ToolsInfo);

			EditorGUI.BeginChangeCheck();

			Handles.BeginGUI();
			{
				_editorUIRect = new Rect(10, screenHeight - 150, screenWidth - 30, 100);

				Rect labelRect = new Rect(10, _editorUIRect.y - 24, screenWidth - 30, 26);
				GUIStyle toolbarStyle = new GUIStyle(EditorStyles.toolbar);
				toolbarStyle.fixedHeight = 30;
				toolbarStyle.fixedWidth = labelRect.width;

				GUILayout.BeginArea(labelRect, toolbarStyle);
				{
					GUIStyle boldLabelStyle = EditorStyles.boldLabel;
					boldLabelStyle.alignment = TextAnchor.UpperLeft;

					using (var hl = new GUILayout.HorizontalScope())
					{
						GUILayout.Label(_toolsLabel, boldLabelStyle);

						int selectedMode = GUILayout.Toolbar((int)_interactionMode, _interactionModeLabels);
						if (selectedMode != (int)_interactionMode)
						{
							SwitchToMode((ToolInteractionMode)selectedMode);
						}
					}
				}
				GUILayout.EndArea();

				GUILayout.BeginArea(_editorUIRect, textBackground);
				{
					using (var horizontalLayout = new GUILayout.HorizontalScope())
					{
						switch(_interactionMode)
						{
							case ToolInteractionMode.VIEW:
							{
								DrawViewModeInfo(asset);
								break;
							}
							case ToolInteractionMode.PAINT:
							{
								DrawPaintModeInfo(asset);
								break;
							}
							case ToolInteractionMode.EDIT:
							{
								DrawEditModeInfo(asset);
								break;
							}
						}
					}
				}
				GUILayout.EndArea();
			}
			Handles.EndGUI();

			if (EditorGUI.EndChangeCheck())
			{
				_toolsInfoSerializedObject.ApplyModifiedProperties();
			}
		}

		private void SwitchToMode(ToolInteractionMode newMode)
		{
			_interactionMode = newMode;

			if(_interactionMode == ToolInteractionMode.VIEW)
			{
				// Hide the editable mesh render and collider
				//foreach(HEU_AttributesStore attributesStore in _attributesStores)
				//{
				//	attributesStore.DisableEditMode();
				//}

				if(_selectedAttributesStore != null)
				{
					_selectedAttributesStore.DisableEditMode();
				}
			}
			else if (_interactionMode == ToolInteractionMode.PAINT)
			{

			}
			else if (_interactionMode == ToolInteractionMode.EDIT)
			{

			}
		}

		private void DrawViewModeInfo(HEU_HoudiniAsset asset)
		{

		}

		private void DrawPaintModeInfo(HEU_HoudiniAsset asset)
		{
			using (var verticalSpace = new GUILayout.VerticalScope(EditorStyles.helpBox))
			{
				// Tool Settings

				HEU_EditorUtility.EditorDrawFloatProperty(_toolsInfoSerializedObject, "_paintBrushSize", "Brush Size");
				HEU_EditorUtility.EditorDrawFloatProperty(_toolsInfoSerializedObject, "_paintBrushRate", "Brush Rate");

				HEU_EditorUtility.EditorDrawSerializedProperty(_toolsInfoSerializedObject, "_brushHandleColor", "Handle Color");

				if(GUILayout.Button("Fill"))
				{
					// TODO: fill with selected paint value
				}
			}

			using (var verticalSpace = new GUILayout.VerticalScope(EditorStyles.helpBox))
			{
				// Attribute Selection, and paint values

				DrawAttributeSelection(asset);
			}

			Event currentEvent = Event.current;
			int controlID = GUIUtility.GetControlID(FocusType.Keyboard);
			EventType eventType = currentEvent.GetTypeForControl(controlID);

			switch (eventType)
			{
				case EventType.MouseDown:
				{
					if (currentEvent.button == 0)
					{
						currentEvent.Use();
					}
					break;
				}
				case EventType.MouseUp:
				{
					if (currentEvent.button == 0)
					{
						currentEvent.Use();
					}
					break;
				}
				case EventType.Layout:
				{
					// This disables deselection on asset while in Add mode
					HandleUtility.AddDefaultControl(GUIUtility.GetControlID(FocusType.Passive));

					break;
				}
			}
		}

		private void DrawAttributeSelection(HEU_HoudiniAsset asset)
		{
			// Try to re-use the last selected node
			string lastSelectedNodeName = null;
			SerializedProperty lastSelectedNodeNameProperty = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_lastAttributeNodeName");
			if (lastSelectedNodeNameProperty != null)
			{
				lastSelectedNodeName = lastSelectedNodeNameProperty.stringValue;
			}

			// Get the names of the all editable nodes having an attribute store for this asset
			// While doing that, we'll find the last selected attribute store
			int lastSelectedIndex = 0;
			List<string> nodeNames = new List<string>();
			foreach (HEU_AttributesStore attributeStore in _attributesStores)
			{
				HEU_GeoNode geoNode = asset.GetInternalGeoNode(attributeStore.GeoID);
				if (geoNode != null)
				{
					HEU_PartData partData = geoNode.GetPartFromPartID(attributeStore.PartID);
					if (partData != null)
					{
						string partName = partData.PartName;
						if (!nodeNames.Contains(partName))
						{
							nodeNames.Add(partName);

							if (_selectedAttributesStore == null)
							{
								// Either re-select last selected node, or select the first one found
								if (string.IsNullOrEmpty(lastSelectedNodeName) || lastSelectedNodeName.Equals(partName))
								{
									lastSelectedNodeName = partName;
									lastSelectedIndex = nodeNames.Count - 1;
									SetSelectedAttributeStore(attributeStore);
								}
							}
						}
					}
				}
			}

			// Try to re-use the last selected attribute
			string lastSelectedAttributeName = null;
			SerializedProperty lastSelectedAttributeProperty = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_lastAttributeName");
			if (lastSelectedAttributeProperty != null)
			{
				lastSelectedAttributeName = lastSelectedAttributeProperty.stringValue;
			}

			// Display a dropdown list of editable nodes with attribute stores
			int currentSelectedIndex = EditorGUILayout.Popup("Editable Node", lastSelectedIndex, nodeNames.ToArray());
			if (currentSelectedIndex != lastSelectedIndex)
			{
				// User changed node selection, so update it
				lastSelectedNodeName = nodeNames[currentSelectedIndex];
				lastSelectedNodeNameProperty.stringValue = lastSelectedNodeName;

				SetSelectedAttributeStore(null);

				foreach (HEU_AttributesStore attributeStore in _attributesStores)
				{
					HEU_GeoNode geoNode = asset.GetInternalGeoNode(attributeStore.GeoID);
					if (geoNode != null)
					{
						HEU_PartData partData = geoNode.GetPartFromPartID(attributeStore.PartID);
						if (partData != null)
						{
							string partName = partData.PartName;
							if (partName.Equals(lastSelectedNodeName))
							{
								SetSelectedAttributeStore(attributeStore);
								break;
							}
						}
					}
				}

				// Reset selected attribute to first attribute
				SetSelectedAttributeData(_selectedAttributesStore.GetAttributeData(0));
				lastSelectedAttributeName = _selectedAttributeData._name;
				lastSelectedAttributeProperty.stringValue = lastSelectedAttributeName;
			}
			else
			{
				// Since selected node hasn't changed, re-use the last selected attribute
				SetSelectedAttributeData(_selectedAttributesStore.GetAttributeData(lastSelectedAttributeName));
			}

			// Get attribute names for selected node
			List<string> attributeNames = _selectedAttributesStore.GetAttributeNames();

			// Find the last selected attribute index
			int lastAttributeIndex = -1;
			if (!string.IsNullOrEmpty(lastSelectedAttributeName))
			{
				for(int i = 0; i < attributeNames.Count; ++i)
				{
					if (lastSelectedAttributeName.Equals(attributeNames[i]))
					{
						lastAttributeIndex = i;
						break;
					}
				}
			}

			// Use first attribute as default if none selected last time
			if(lastAttributeIndex == -1)
			{
				lastAttributeIndex = 0;
				HEU_AttributeData data = _selectedAttributesStore.GetAttributeData(0);
				if (data != null)
				{
					lastSelectedAttributeProperty.stringValue = data._name;
				}
			}

			// Display attributes as dropdown
			if (attributeNames.Count > 0)
			{
				int currentAttributeIndex = EditorGUILayout.Popup("Attribute", lastAttributeIndex, attributeNames.ToArray());
				if (currentAttributeIndex != lastAttributeIndex)
				{
					// User changed attribute selection, so update it
					SetSelectedAttributeData(_selectedAttributesStore.GetAttributeData(attributeNames[currentAttributeIndex]));
					lastSelectedAttributeProperty.stringValue = _selectedAttributeData._name;
				}
			}

			// Now display the value as an editable field
			if(_selectedAttributeData != null)
			{
				if (_selectedAttributeData._attributeType == HEU_AttributeData.AttributeType.INT)
				{
					_selectedToolsValuesProperty = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_paintIntValue");
					if (_selectedToolsValuesProperty != null)
					{
						ResizeSerializedPropertyArray(_selectedToolsValuesProperty, _selectedAttributeData._attributeInfo.tupleSize);
						HEU_EditorUtility.EditorDrawArrayProperty(_selectedToolsValuesProperty, HEU_EditorUtility.EditorDrawIntProperty, _paintValuesLabel);
					}
				}
				else if (_selectedAttributeData._attributeType == HEU_AttributeData.AttributeType.FLOAT)
				{
					_selectedToolsValuesProperty = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_paintFloatValue");
					if (_selectedToolsValuesProperty != null)
					{
						ResizeSerializedPropertyArray(_selectedToolsValuesProperty, _selectedAttributeData._attributeInfo.tupleSize);
						HEU_EditorUtility.EditorDrawArrayProperty(_selectedToolsValuesProperty, HEU_EditorUtility.EditorDrawFloatProperty, _paintValuesLabel);
					}
					
					// Display paint color selector if this is a color attribute
					if(_selectedAttributeData.IsColorAttribute())
					{
						Color color = Color.white;

						if(_selectedToolsValuesProperty.arraySize >= 3)
						{
							color.r = _selectedToolsValuesProperty.GetArrayElementAtIndex(0).floatValue;
							color.g = _selectedToolsValuesProperty.GetArrayElementAtIndex(1).floatValue;
							color.b = _selectedToolsValuesProperty.GetArrayElementAtIndex(2).floatValue;

							if (_selectedToolsValuesProperty.arraySize >= 4)
							{
								color.a = _selectedToolsValuesProperty.GetArrayElementAtIndex(3).floatValue;
							}
						}

						Color newColor = EditorGUILayout.ColorField("Paint Color", color);
						if (color != newColor)
						{
							if (_selectedToolsValuesProperty.arraySize >= 3)
							{
								_selectedToolsValuesProperty.GetArrayElementAtIndex(0).floatValue = newColor.r;
								_selectedToolsValuesProperty.GetArrayElementAtIndex(1).floatValue = newColor.g;
								_selectedToolsValuesProperty.GetArrayElementAtIndex(2).floatValue = newColor.b;

								if (_selectedToolsValuesProperty.arraySize >= 4)
								{
									_selectedToolsValuesProperty.GetArrayElementAtIndex(3).floatValue = newColor.a;
								}
							}
						}
					}
					
				}
				else if (_selectedAttributeData._attributeType == HEU_AttributeData.AttributeType.STRING)
				{
					_selectedToolsValuesProperty = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_paintStringValue");
					if (_selectedToolsValuesProperty != null)
					{
						ResizeSerializedPropertyArray(_selectedToolsValuesProperty, _selectedAttributeData._attributeInfo.tupleSize);
						HEU_EditorUtility.EditorDrawArrayProperty(_selectedToolsValuesProperty, HEU_EditorUtility.EditorDrawTextProperty, _paintValuesLabel);
					}
				}
			}
		}

		private void ResizeSerializedPropertyArray(SerializedProperty arrayProperty, int newSize)
		{
			if(arrayProperty.arraySize != newSize)
			{
				arrayProperty.arraySize = newSize;
				
				_GUIChanged = true;
			}
		}

		private void SetSelectedAttributeStore(HEU_AttributesStore newStore)
		{
			if(_selectedAttributesStore != null)
			{
				_selectedAttributesStore.DisableEditMode();
			}

			_selectedAttributesStore = null;

			if (newStore != null)
			{
				SerializedObject serializedObject = GetOrCreateSerializedAttributesStore(newStore);
				_selectedAttributesStore = serializedObject.targetObject as HEU_AttributesStore;

				if (_selectedAttributesStore != null)
				{
					if (_interactionMode == ToolInteractionMode.PAINT || _interactionMode == ToolInteractionMode.EDIT)
					{
						_selectedAttributesStore.EnableEditMode();
					}
				}
			}
		}

		private void SetSelectedAttributeData(HEU_AttributeData newAttr)
		{
			_selectedAttributeData = newAttr;
		}

		private void DrawEditModeInfo(HEU_HoudiniAsset asset)
		{

		}

		// SCENE DRAWING ----------------------------------------------------------------------------------------------

		public void DrawToolsEditorScene(HEU_HoudiniAsset asset)
		{
			// Resetting the selected attributes which will be set in the DrawSceneInfoPanel 
			// based on user selection
			_selectedAttributesStore = null;
			_selectedAttributeData = null;

			_currentCamera = Camera.current;

			_GUIChanged = false;

			EditorGUI.BeginChangeCheck();

			// Draw the info panel first. This gets the user selected node and attribute.
			DrawSceneInfoPanel(asset);

			if (_interactionMode == ToolInteractionMode.VIEW)
			{
				DrawViewModeScene(asset);
			}
			else if (_interactionMode == ToolInteractionMode.PAINT)
			{
				DrawPaintModeScene(asset);
			}
			else if(_interactionMode == ToolInteractionMode.EDIT)
			{
				DrawEditModeScene(asset);
			}

			// At end of drawing, we disable the selected attribute store's edit state
			// so that the mesh render and collider will be disabled
			//SetSelectedAttributeStore(null);
			//_selectedAttributeData = null;

			if (EditorGUI.EndChangeCheck() || _GUIChanged)
			{
				// Presume that if a serialized object was cached, then most likely it was modified
				foreach (var attributeStoresPair in _serializedAttributesStoresCache)
				{
					attributeStoresPair.Value.ApplyModifiedProperties();
				}

				_toolsInfoSerializedObject.ApplyModifiedProperties();
			}
		}

		private void DrawViewModeScene(HEU_HoudiniAsset asset)
		{

		}

		private void DrawPaintModeScene(HEU_HoudiniAsset asset)
		{
			if(_selectedAttributesStore == null || _selectedAttributeData == null)
			{
				// Nothing to do if no attribute selected
				return;
			}

			Event currentEvent = Event.current;

			int controlID = GUIUtility.GetControlID(FocusType.Keyboard);
			//EventType eventType = currentEvent.GetTypeForControl(controlID);

			//KeyCode pressedKey = GetPressedKey(currentEvent);

			//bool bMouseDown = currentEvent.type == EventType.MouseDown;
			//bool bMouseDown = eventType == EventType.MouseDown;
			Vector3 mousePosition = HEU_EditorUI.GetMousePosition(ref currentEvent, _currentCamera);

			// Enable the mesh collider so that we can raycast to paint using brush
			MeshCollider collider = _selectedAttributesStore.OutputMeshCollider;
			if (!collider.enabled)
			{
				collider.enabled = true;
			}

			float brushRadius = GetBrushRadius();

			// TODO: use Handles.DrawingScope
			Color originalHandleColor = Handles.color;
			Color newHandleColor = originalHandleColor;
			
			SerializedProperty handleColorProperty = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_brushHandleColor");
			if(handleColorProperty != null)
			{
				newHandleColor = handleColorProperty.colorValue;
			}

			Ray ray = _currentCamera.ScreenPointToRay(mousePosition);
			ray.origin = _currentCamera.transform.position;

			RaycastHit hit;
			if(collider.Raycast(ray, out hit, _intersectionRayLength))
			{
				if(currentEvent.type == EventType.ScrollWheel)
				{
					// Brush resize
					brushRadius -= currentEvent.delta.y * _mouseWheelBrushSizeMultiplier;
					brushRadius = UpdateBrushSize(brushRadius);

					_GUIChanged = true;
					currentEvent.Use();
				}
				else if (currentEvent.type == EventType.MouseDrag && currentEvent.shift)
				{
					// Brush resize
					brushRadius += currentEvent.delta.x * _mouseWheelBrushSizeMultiplier;
					brushRadius = UpdateBrushSize(brushRadius);

					_GUIChanged = true;
					currentEvent.Use();
				}
				else if(!currentEvent.shift && (currentEvent.type == EventType.MouseDown || currentEvent.type == EventType.MouseDrag))
				{
					// Do Paint
					HandlePaintEvent(hit, brushRadius);
				}

				Handles.color = newHandleColor;
				Vector3 endPt = hit.point + (Vector3.Normalize(hit.normal) * brushRadius);
				Handles.DrawAAPolyLine(2f, new Vector3[] { hit.point, endPt });

				HEU_EditorUI.DrawCircleCap(controlID, hit.point, Quaternion.FromToRotation(Vector3.forward, hit.normal), brushRadius);
			}

			Handles.color = originalHandleColor;
		}

		private void DrawEditModeScene(HEU_HoudiniAsset asset)
		{

		}

		private float GetBrushRadius()
		{
			SerializedProperty property = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_paintBrushSize");
			return (property != null) ? property.floatValue : 1f;
		}

		private float UpdateBrushSize(float radius)
		{
			if(radius < 0.01f)
			{
				radius = 0.01f;
			}

			SerializedProperty property = HEU_EditorUtility.GetSerializedProperty(_toolsInfoSerializedObject, "_paintBrushSize");
			if(property != null && (Mathf.Abs(property.floatValue - radius) >= float.Epsilon))
			{
				property.floatValue = radius;
			}

			return radius;
		}

		private KeyCode GetPressedKey(Event currentEvent)
		{
			if(currentEvent.type == EventType.KeyDown)
			{
				return currentEvent.keyCode;
			}
			return KeyCode.None;
		}

		private void HandlePaintEvent(RaycastHit hit, float brushRadius)
		{
			if(_selectedAttributesStore == null || _selectedAttributeData == null || _selectedToolsValuesProperty == null)
			{
				return;
			}

			HEU_AttributesStore.SetAttributeValues setAttrFunc = null;
			if (_selectedAttributeData._attributeType == HEU_AttributeData.AttributeType.INT)
			{
				setAttrFunc = HEU_AttributesStore.SetAttributeValueInt;
			}
			else if (_selectedAttributeData._attributeType == HEU_AttributeData.AttributeType.FLOAT)
			{
				setAttrFunc = HEU_AttributesStore.SetAttributeValueFloat;
			}
			else if (_selectedAttributeData._attributeType == HEU_AttributeData.AttributeType.STRING)
			{
				setAttrFunc = HEU_AttributesStore.SetAttributeValueString;
			}
			else
			{
				// Unsupported type
				return;
			}

			HEU_ToolsInfo toolsInfo = _toolsInfoSerializedObject.targetObject as HEU_ToolsInfo;

			Mesh mesh = _selectedAttributesStore.OutputMesh;
			if (mesh != null)
			{
				Vector3 localHitPoint = _selectedAttributesStore.OutputTransform.InverseTransformPoint(hit.point);

				int vertexCount = mesh.vertexCount;
				Vector3[] vertices = mesh.vertices;

				for (int i = 0; i < vertexCount; ++i)
				{
					if (Vector3.Distance(localHitPoint, vertices[i]) <= brushRadius)
					{
						_selectedAttributesStore.PaintAttribute(_selectedAttributeData, toolsInfo, i, 1f, setAttrFunc);
					}
				}

				// Flag that attribute needs uploading
			}
		}

		
	}

}   // HoudiniEngineUnity