﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using System.Collections.Generic;
using System.Text;

namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;
	using HAPI_ParmId = System.Int32;
	using HAPI_StringHandle = System.Int32;


	/// <summary>
	/// Contains data and logic for curve node drawing and editing.
	/// </summary>
	public class HEU_Curve : ScriptableObject
	{
		// DATA -------------------------------------------------------------------------------------------------------

		[SerializeField]
		private HAPI_NodeId _geoID;

		[SerializeField]
		private List<Vector3> _points = new List<Vector3>();

		[SerializeField]
		private Vector3[] _vertices;

		[SerializeField]
		private bool _isEditable;

		public bool IsEditable() { return _isEditable; }

		[SerializeField]
		private HEU_Parameters _parameters;

		public HEU_Parameters Parameters { get { return _parameters; } }

		[SerializeField]
		private bool _bUploadParameterPreset;

		public void SetUploadParameterPreset(bool bValue) { _bUploadParameterPreset = bValue; }

		[SerializeField]
		private string _curveName;

		public string CurveName { get { return _curveName; } }

		public GameObject _targetGameObject;

		[SerializeField]
		private bool _isGeoCurve;

		public bool IsGeoCurve() { return _isGeoCurve; }


		public enum CurveEditState
		{
			INVALID,
			GENERATED,
			EDITING,
			REQUIRES_GENERATION
		}
		[SerializeField]
		private CurveEditState _editState;

		public CurveEditState EditState { get { return _editState; } }

		// Types of interaction with this curve. Used by Editor.
		public enum Interaction
		{
			VIEW,
			ADD,
			EDIT
		}

		// Preferred interaction mode when this a curve selected. Allows for quick access for curve editing.
		public static Interaction PreferredNextInteractionMode = Interaction.VIEW;

		public enum CurveDrawCollision
		{
			COLLIDERS,
			LAYERMASK
		}


		// LOGIC ------------------------------------------------------------------------------------------------------

		public static HEU_Curve CreateSetupCurve(HEU_HoudiniAsset parentAsset, bool isEditable, string curveName, HAPI_NodeId geoID, bool bGeoCurve)
		{
			HEU_Curve newCurve = ScriptableObject.CreateInstance<HEU_Curve>();
			newCurve._isEditable = isEditable;
			newCurve._curveName = curveName;
			newCurve._geoID = geoID;
			newCurve.SetEditState(CurveEditState.INVALID);
			newCurve._isGeoCurve = bGeoCurve;
			parentAsset.AddCurve(newCurve);
			return newCurve;
		}

		public void DestroyAllData()
		{
			if (_parameters != null)
			{
				_parameters.CleanUp();
				_parameters = null;
			}

			if(_isGeoCurve && _targetGameObject != null)
			{
				HEU_HAPIUtility.DestroyGameObject(_targetGameObject);
				_targetGameObject = null;
			}
		}

		public void SetCurveName(string name)
		{
			_curveName = name;
			if(_targetGameObject != null)
			{
				_targetGameObject.name = name;
			}
		}

		public void UploadParameterPreset(HEU_SessionBase session, HAPI_NodeId geoID, HEU_HoudiniAsset parentAsset)
		{
			// TODO FIXME
			// This fixes up the geo IDs for curves, and upload parameter values to Houdini.
			// This is required for curves in saved scenes, as its parameter data is not part of the parent asset's
			// parameter preset. Also the _geoID and parameters._nodeID could be different so uploading the
			// parameter values before cooking would not be valid for those IDs. This waits until after cooking
			// to then upload and cook just the curve.
			// Admittedly this is a temporary solution until a proper workaround is in place. Ideally for an asset reload
			// the object node and geo node names can be used to match up the IDs and then parameter upload can happen
			// before cooking.

			_geoID = geoID;

			if(_parameters != null)
			{
				_parameters._nodeID = geoID;

				if(_bUploadParameterPreset)
				{
					_parameters.UploadPresetData(session);
					_parameters.UploadValuesToHoudini(session, parentAsset);

					bool bResult = HEU_HAPIUtility.CookNodeInHoudini(session, geoID, false, _curveName);

					_bUploadParameterPreset = false;
				}
			}
		}

		public bool UpdateCurve(HEU_SessionBase session, HAPI_PartId partID)
		{
			HAPI_PartInfo partInfo = new HAPI_PartInfo();
			bool bResult = session.GetPartInfo(_geoID, partID, ref partInfo);
			if (!bResult)
			{
				Debug.LogErrorFormat("Unable to get PartInfo for geo node {0} and part {1}.", _geoID, partID);
				return false;
			}

			int vertexCount = 0;

			// Get position attributes.
			// Note that for an empty curve (ie. no position attributes) this query will fail, 
			// but the curve is still valid, so we simply set to null vertices. This allows 
			// user to add points later on.
			HAPI_AttributeInfo posAttrInfo = new HAPI_AttributeInfo();
			float[] posAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, _geoID, partID, HEU_Defines.HAPI_ATTRIB_POSITION, ref posAttrInfo, ref posAttr, session.GetAttributeFloatData);
			if (posAttrInfo.exists)
			{
				vertexCount = posAttrInfo.count;
			}

			// Curve guides from position attributes
			_vertices = new Vector3[vertexCount];
			for(int i = 0; i < vertexCount; ++i)
			{
				_vertices[i][0] = -posAttr[i * 3 + 0];
				_vertices[i][1] =  posAttr[i * 3 + 1];
				_vertices[i][2] =  posAttr[i * 3 + 2];
			}

			return true;
		}

		public void GenerateMesh(GameObject inGameObject)
		{
			_targetGameObject = inGameObject;

			MeshFilter meshFilter = _targetGameObject.GetComponent<MeshFilter>();
			if(meshFilter == null)
			{
				meshFilter = _targetGameObject.AddComponent<MeshFilter>();
			}

			MeshRenderer meshRenderer = _targetGameObject.GetComponent<MeshRenderer>();
			if(meshRenderer == null)
			{
				meshRenderer = _targetGameObject.AddComponent<MeshRenderer>();

				Shader shader = HEU_MaterialFactory.FindPluginShader("LineShader");
				meshRenderer.sharedMaterial = new Material(shader);
				meshRenderer.sharedMaterial.SetColor("_Color", HEU_PluginSettings.LineColor);
			}

			Mesh mesh = meshFilter.sharedMesh;

			if(_points.Count <= 1)
			{
				if (mesh != null)
				{
					mesh.Clear();
					mesh = null;
				}
			}
			else
			{
				if (mesh == null)
				{
					mesh = new Mesh();
					mesh.name = "Curve";
				}

				int[] indices = new int[_vertices.Length];
				for(int i = 0; i < _vertices.Length; ++i)
				{
					indices[i] = i;
				}

				mesh.Clear();
				mesh.vertices = _vertices;
				mesh.SetIndices(indices, MeshTopology.LineStrip, 0);
				mesh.RecalculateBounds();

				mesh.UploadMeshData(false);
			}

			meshFilter.sharedMesh = mesh;
			meshRenderer.enabled = HEU_PluginSettings.Curves_ShowInSceneView;

			SetEditState(CurveEditState.GENERATED);
		}

		public void SyncFromParameters(HEU_SessionBase session, HEU_HoudiniAsset parentAsset)
		{
			HAPI_NodeInfo geoNodeInfo = new HAPI_NodeInfo();
			if (!session.GetNodeInfo(_geoID, ref geoNodeInfo))
			{
				return;
			}

			if (_parameters != null)
			{
				_parameters.CleanUp();
			}
			else
			{
				_parameters = ScriptableObject.CreateInstance<HEU_Parameters>();
			}

			string geoNodeName = HEU_SessionManager.GetString(geoNodeInfo.nameSH, session);
			_parameters._uiLabel = geoNodeName.ToUpper() + " PARAMETERS";

			bool bResult = _parameters.Initialize(session, _geoID, ref geoNodeInfo, null, null, parentAsset);
			if (!bResult)
			{
				Debug.LogWarningFormat("Parameter generate failed for geo node {0}.", geoNodeInfo.id);
				_parameters.CleanUp();
				return;
			}

			_points.Clear();
			string pointList = _parameters.GetStringFromParameter(HEU_Defines.CURVE_COORDS_PARAM);
			if(!string.IsNullOrEmpty(pointList))
			{
				string[] pointSplit = pointList.Split(' ');
				foreach (string str in pointSplit)
				{
					string[] vecSplit = str.Split(',');
					if (vecSplit.Length == 3)
					{
						_points.Add(new Vector3(-System.Convert.ToSingle(vecSplit[0]),
							System.Convert.ToSingle(vecSplit[1]),
							System.Convert.ToSingle(vecSplit[2])));
					}
				}
			}

			// Since we just reset / created new our parameters and sync'd, we also need to 
			// get the preset from Houdini session
			if (!HEU_EditorUtility.IsEditorPlaying() && IsEditable())
			{
				DownloadPresetData(session);
			}
		}

		/// <summary>
		/// Returns points array as string
		/// </summary>
		/// <param name="points">List of points to stringify</param>
		/// <returns></returns>
		public static string GetPointsString(List<Vector3> points)
		{
			StringBuilder sb = new StringBuilder();
			foreach (Vector3 pt in points)
			{
				sb.AppendFormat("{0},{1},{2} ", -pt[0], pt[1], pt[2]);
			}
			return sb.ToString();
		}

		public void SetEditState(CurveEditState editState)
		{
			_editState = editState;
		}

		public void SetCurvePoint(int pointIndex, Vector3 newPosition)
		{
			if(pointIndex >= 0 && pointIndex < _points.Count)
			{
				_points[pointIndex] = newPosition;
			}
		}

		public Vector3 GetCurvePoint(int pointIndex)
		{
			if(pointIndex >= 0 && pointIndex < _points.Count)
			{
				return _points[pointIndex];
			}
			return Vector3.zero;
		}

		public List<Vector3> GetAllPoints()
		{
			return _points;
		}

		public int GetNumPoints()
		{
			return _points.Count;
		}

		public Vector3 GetTransformedPoint(int pointIndex)
		{
			if(pointIndex >= 0 && pointIndex < _points.Count)
			{
				return GetTransformedPosition(_points[pointIndex]);
			}
			return Vector3.zero;
		}

		public Vector3 GetTransformedPosition(Vector3 inPosition)
		{
			return this._targetGameObject.transform.TransformPoint(inPosition);
		}

		public Vector3 GetInvertedTransformedPosition(Vector3 inPosition)
		{
			return this._targetGameObject.transform.InverseTransformPoint(inPosition);
		}

		public Vector3[] GetVertices()
		{
			return _vertices;
		}

		public void SetCurveGeometryVisibility(bool bVisible)
		{
			if(_targetGameObject != null)
			{
				MeshRenderer renderer = _targetGameObject.GetComponent<MeshRenderer>();
				if(renderer != null)
				{
					renderer.enabled = bVisible;
				}
			}
		}

		public void DownloadPresetData(HEU_SessionBase session)
		{
			if(_parameters != null)
			{
				_parameters.DownloadPresetData(session);
			}
		}

		public void UploadPresetData(HEU_SessionBase session)
		{
			if (_parameters != null)
			{
				_parameters.UploadPresetData(session);
			}
		}
	}

}   // HoudiniEngineUnity