﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

// Uncomment to profile
//#define HEU_PROFILER_ON

using System;
using System.Text;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;


	/// <summary>
	/// Represents a Geometry (SOP) node.
	/// </summary>
	public class HEU_GeoNode : ScriptableObject
	{
		//	DATA ------------------------------------------------------------------------------------------------------

		public HAPI_NodeId GeoID { get { return _geoInfo.nodeId; } }

		[SerializeField]
		private HAPI_GeoInfo _geoInfo;

		[SerializeField]
		private string _geoName;
		public string GeoName { get { return _geoName; } }

		public HAPI_GeoType GeoType { get { return _geoInfo.type; } }

		public bool Editable { get { return _geoInfo.isEditable; } }

		public bool Displayable { get { return _geoInfo.isDisplayGeo; } }

		public bool IsVisible() { return _containerObjectNode.IsVisible(); }

		public bool IsIntermediateOrEditable() { return (_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_INTERMEDIATE || (_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_DEFAULT && _geoInfo.isEditable)); }

		[SerializeField]
		private List<HEU_PartData> _parts;

		[SerializeField]
		private HEU_ObjectNode _containerObjectNode;

		public HEU_HoudiniAsset ParentAsset { get { return (_containerObjectNode != null) ? _containerObjectNode.ParentAsset : null; } }

		[SerializeField]
		private HEU_InputNode _inputNode;

		[SerializeField]
		private HEU_Curve _geoCurve;


		//  LOGIC -----------------------------------------------------------------------------------------------------

		public HEU_GeoNode()
		{
			Reset();
		}

		/// <summary>
		/// Destroy all generated data.
		/// </summary>
		public void DestroyAllData()
		{
			DestroyParts(_parts);

			if(_inputNode != null)
			{
				HEU_SessionBase session = null;
				if (ParentAsset != null)
				{
					ParentAsset.RemoveInputNode(_inputNode);
					session = ParentAsset.GetAssetSession(false);
				}

				_inputNode.DestroyAllData(session);
				HEU_GeneralUtility.DestroyImmediate(_inputNode);
				_inputNode = null;
			}

			if (_geoCurve != null)
			{
				if (ParentAsset != null)
				{
					ParentAsset.RemoveCurve(_geoCurve);
				}
				_geoCurve.DestroyAllData();
				HEU_GeneralUtility.DestroyImmediate(_geoCurve);
				_geoCurve = null;
			}
		}

		/// <summary>
		/// Destroy parts and their data.
		/// </summary>
		/// <param name="bRegisterUndo">Register Undo action</param>
		private static void DestroyParts(List<HEU_PartData> parts)
		{
			int numParts = parts.Count;
			for (int i = 0; i < numParts; ++i)
			{
				DestroyPart(parts[i]);
			}
			parts.Clear();
		}

		private static void DestroyPart(HEU_PartData part)
		{
			part.DestroyAllData();
			HEU_GeneralUtility.DestroyImmediate(part);
		}

		/// <summary>
		/// Destroy the generated mesh data.
		/// </summary>
		/// <param name="bRegisterUndo">Register Undo action</param>
		public void DestroyGeneratedMeshData(bool bRegisterUndo)
		{
			int numParts = _parts.Count;
			for (int i = 0; i < numParts; ++i)
			{
				_parts[i].DestroyGeneratedMeshData(bRegisterUndo: bRegisterUndo);
			}
		}

		public void Reset()
		{
			_geoName = "";

			_geoInfo = new HAPI_GeoInfo();
			_geoInfo.nodeId = -1;
			_geoInfo.type = HAPI_GeoType.HAPI_GEOTYPE_DEFAULT;
			_parts = new List<HEU_PartData>();
		}

		public void Initialize(HEU_SessionBase session, HAPI_GeoInfo geoInfo, HEU_ObjectNode containerObjectNode)
		{
			_containerObjectNode = containerObjectNode;
			_geoInfo = geoInfo;
			_geoName = HEU_SessionManager.GetString(_geoInfo.nameSH, session);

			//Debug.Log(string.Format("GeoNode initialized with ID={0}, name={1}, type={2}", GeoID, GeoName, geoInfo.type));
		}

		public bool DoesThisRequirePotentialCook()
		{
			if((_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_INPUT) 
				|| (_geoInfo.isTemplated && !HEU_PluginSettings.CookTemplatedGeos && !_geoInfo.isEditable)
				|| (!_geoInfo.hasGeoChanged)
				|| (!_geoInfo.isDisplayGeo && (_geoInfo.type != HAPI_GeoType.HAPI_GEOTYPE_CURVE)))
			{
				return false;
			}
			return true;
		}

		public void UpdateGeo(HEU_SessionBase session)
		{
			// Create or recreate parts.

			bool bObjectInstancer = _containerObjectNode.IsInstancer();

			// Save list of old parts. We'll destroy these after creating new parts.
			// The reason for temporarily keeping these is to transfer data (eg. instance overrides, attribute data)
			List<HEU_PartData> oldParts = new List<HEU_PartData>(_parts);
			_parts.Clear();

			try
			{
				// TODO: FIXME: When a heightfield erode node is in the network, it causes multiple display + editable
				// nodes to appear. This is a hack to reduce those down to just the single display node. But should be fixed properly!
				//if (!_geoInfo.isDisplayGeo || (_geoInfo.isDisplayGeo && _geoInfo.isEditable && _geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_DEFAULT))

				if(!_geoInfo.isDisplayGeo)
				{
					if(ParentAsset.IgnoreNonDisplayNodes)
					{
						return;
					}
					else if (!_geoInfo.isEditable 
							|| (_geoInfo.type != HAPI_GeoType.HAPI_GEOTYPE_DEFAULT 
								&& _geoInfo.type != HAPI_GeoType.HAPI_GEOTYPE_INTERMEDIATE
								&& _geoInfo.type != HAPI_GeoType.HAPI_GEOTYPE_CURVE))
					{
						return;
					}
				}

				if (_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_CURVE)
				{
					ProcessGeoCurve(session);
				}
				else
				{
					int numParts = _geoInfo.partCount;
					//Debug.Log("Number of parts: " + numParts);
					//Debug.LogFormat("GeoNode type {0}, isTemplated: {1}, isDisplayGeo: {2}, isEditable: {3}", _geoInfo.type, _geoInfo.isTemplated, _geoInfo.isDisplayGeo, _geoInfo.isEditable);
					for (int i = 0; i < numParts; ++i)
					{
						HAPI_PartInfo partInfo = new HAPI_PartInfo();
						if (!session.GetPartInfo(GeoID, i, ref partInfo))
						{
							Debug.LogErrorFormat("Unable to get PartInfo for geo node {0} and part {1}.", GeoID, i);
							continue;
						}

						// Find the old part for this new part.
						HEU_PartData part = null;
						HEU_PartData oldMatchedPart = null;

						foreach (HEU_PartData oldPart in oldParts)
						{
							string partName = HEU_SessionManager.GetString(partInfo.nameSH, session);
							if (oldPart.PartName.Equals(partName))
							{
								oldMatchedPart = oldPart;
							}
						}

						if (oldMatchedPart != null)
						{
							//Debug.Log("Found matched part: " + oldMatchedPart.name);

							List<HEU_ObjectInstanceInfo> sourceObjectInstanceInfos = null;
							if (bObjectInstancer)
							{
								// ProcessPart will clear out the object instances, so hence why
								// we keep a copy here, then restore after processing the parts.
								sourceObjectInstanceInfos = oldMatchedPart.GetObjectInstanceInfos();
							}

							// Clear out old generated data
							oldMatchedPart.ClearGeneratedData();

							part = oldMatchedPart;
							oldParts.Remove(oldMatchedPart);

							ProcessPart(session, i, ref partInfo, ref part);

							if (part != null && bObjectInstancer && sourceObjectInstanceInfos != null)
							{
								// Set object instances from old part into new. This keeps the user set object inputs around.
								part.SetObjectInstanceInfos(sourceObjectInstanceInfos);
							}
						}
						else
						{
							ProcessPart(session, i, ref partInfo, ref part);
						}
					}

					ProcessInstancerParts(session);
				}
			}
			finally
			{
				DestroyParts(oldParts);
			}
		}

		/// <summary>
		/// Process custom attribute with Unity script name, and attach any scripts found.
		/// </summary>
		/// <param name="session">Session to use</param>
		public void ProcessUnityScriptAttribute(HEU_SessionBase session)
		{
			if(_parts == null || _parts.Count == 0)
			{
				return;
			}

			HAPI_AttributeInfo scriptAttributeInfo = new HAPI_AttributeInfo();
			int[] scriptAttr = new int[0];

			HEU_GeneralUtility.GetAttribute(session, GeoID, 0, HEU_PluginSettings.UnityScriptAttributeName, ref scriptAttributeInfo, ref scriptAttr, session.GetAttributeStringData);
			if (scriptAttributeInfo.exists)
			{
				if (scriptAttributeInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_DETAIL)
				{
					Debug.LogWarningFormat("Houdini Engine for Unity only supports {0} as detail attributes!", HEU_PluginSettings.UnityScriptAttributeName);
				}
				else if (scriptAttr.Length > 0)
				{
					string scriptToAttach = HEU_SessionManager.GetString(scriptAttr[0]);
					AttachScriptWithInvokeFunction(scriptToAttach);
				}
			}
		}

		/// <summary>
		/// Attach a script in Unity to the asset root gameobject, and optionally
		/// invoke a function with an optional argument.
		/// </summary>
		/// <param name="scriptToAttach">A string with format: scriptname:function:msg</param>
		public void AttachScriptWithInvokeFunction(string scriptToAttach)
		{
			// Script will be attached to the asset root.
			// Then if set, te function will be invoked on the script passing in the message.
			// Format: scriptname:function:msg
			string expectedFormat = "scriptname:function:argument";

			int scriptColon = scriptToAttach.IndexOf(":");
			if(scriptColon <= 0)
			{
				Debug.LogFormat("Invalid script attribute value. Expected format {0}, but got {1}", expectedFormat, scriptToAttach);
				return;
			}

			string scriptTypeName = scriptToAttach.Substring(0, scriptColon).Trim();
			System.Type scriptType = HEU_GeneralUtility.GetSystemTypeByName(scriptTypeName);
			if(scriptType == null)
			{
				Debug.LogFormat("Script with name {0} not found! Unable to attach script from attribute: {1}", scriptTypeName, scriptToAttach);
				return;
			}

			GameObject rootGameObject = ParentAsset.RootGameObject;

			Component component = rootGameObject.GetComponent(scriptType);
			if (component == null)
			{
				Debug.LogFormat("Attaching script {0} to root gameobject", scriptType);
				component = rootGameObject.AddComponent(scriptType);
				if (component == null)
				{
					Debug.LogFormat("Unable to attach script component with type '{0}' from script attribute: {1}", scriptType.ToString(), scriptToAttach);
					return;
				}
			}

			if(scriptColon + 1 >= scriptToAttach.Length)
			{
				// No function
				return;
			}

			int functionColon = scriptToAttach.IndexOf(":", scriptColon + 1);
			int functionNameLength = functionColon - (scriptColon + 1);
			if(functionNameLength > 0)
			{
				string scriptFunction = scriptToAttach.Substring(scriptColon + 1, functionNameLength).Trim();

				if (functionColon + 1 < scriptToAttach.Length)
				{
					// Get argument
					string scriptArgument = scriptToAttach.Substring(functionColon + 1).Trim();
					//Debug.LogFormat("Invoking script function {0} with argument {1}", scriptFunction, scriptArgument);
					component.SendMessage(scriptFunction, scriptArgument, SendMessageOptions.DontRequireReceiver);
				}
				else
				{
					// No argument
					//Debug.LogFormat("Invoking script function {0}", scriptFunction);
					component.SendMessage(scriptFunction, SendMessageOptions.DontRequireReceiver);
				}
			}
		}

		/// <summary>
		/// Process the part at the given index, creating its data (geometry),
		/// and adding it to the list of parts.
		/// </summary>
		/// <param name="session"></param>
		/// <param name="partID"></param>
		/// <returns>A valid HEU_PartData if it has been successfully processed.</returns>
		private void ProcessPart(HEU_SessionBase session, int partID, ref HAPI_PartInfo partInfo, ref HEU_PartData partData)
		{
			HEU_HoudiniAsset parentAsset = ParentAsset;
			bool bResult = true;
			//Debug.LogFormat("Part: name={0}, id={1}, type={2}, instanced={3}, instance count={4}, instance part count={5}", HEU_SessionManager.GetString(partInfo.nameSH, session), partID, partInfo.type, partInfo.isInstanced, partInfo.instanceCount, partInfo.instancedPartCount);

#if HEU_PROFILER_ON
			float processPartStartTime = Time.realtimeSinceStartup;
#endif

			bool isGeoInputType = (_geoInfo.isEditable && _geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_INPUT);
			bool isPartEditable = (_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_INTERMEDIATE || (_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_DEFAULT && _geoInfo.isEditable));

			if (isGeoInputType)
			{
				// Setup for input node to accept inputs
				if (_inputNode == null)
				{
					string partName = HEU_SessionManager.GetString(partInfo.nameSH, session);
					_inputNode = HEU_InputNode.CreateSetupInput(GeoID, 0, partName, HEU_InputNode.InputNodeType.NODE, ParentAsset);
					if(_inputNode != null)
					{
						ParentAsset.AddInputNode(_inputNode);
					}
				}
			}

			if(partInfo.type == HAPI_PartType.HAPI_PARTTYPE_INVALID)
			{
				// Clean up invalid parts
				if (partData != null)
				{
					DestroyPart(partData);
					partData = null;
				}
			}
			else if (partInfo.type < HAPI_PartType.HAPI_PARTTYPE_MAX)
			{
				// Process the part based on type. Keep or ignore.

				// We treat parts of type curve as curves, along with geo nodes that are editable and type curves
				if (partInfo.type == HAPI_PartType.HAPI_PARTTYPE_CURVE)
				{
					if(partData == null)
					{
						partData = ScriptableObject.CreateInstance<HEU_PartData>();
					}

					partData.Initialize(session, partID, GeoID, _containerObjectNode.ObjectID, this, ref partInfo, HEU_PartData.PartOutputType.CURVE, isPartEditable);
					SetupGameObjectAndTransform(partData, parentAsset);
					bResult = partData.ProcessCurvePart(session, ref partInfo);

					// When a Curve asset is used as input node, it creates this editable and useless curve part type.
					// For now deleting it as it causes issues on recook (from scene load), as well as unnecessary curve editor UI.
					// Should revisit sometime in the future to review this.
					if (bResult)
					{
						HEU_Curve curve = partData.GetCurve(false);
						if(curve == null || curve.GetNumPoints() == 0)
						{
							bResult = false;
						}
					}

					if (!bResult)
					{
						DestroyPart(partData);
						partData = null;
					}
				}
				else if (partInfo.type == HAPI_PartType.HAPI_PARTTYPE_VOLUME)
				{
					// We only process "height" volume parts. Other volume parts are ignored for now.

					HAPI_VolumeInfo volumeInfo = new HAPI_VolumeInfo();
					bResult = session.GetVolumeInfo(GeoID, partID, ref volumeInfo);
					if (!bResult)
					{
						Debug.LogErrorFormat("Unable to get volume info for geo node {0} and part {1} ", GeoID, partID);
					}
					else
					{
						string volumeName = HEU_SessionManager.GetString(volumeInfo.nameSH, session);
						if (volumeName.Equals("height"))
						{
							if (partData == null)
							{
								partData = ScriptableObject.CreateInstance<HEU_PartData>();
							}

							partData.Initialize(session, partID, GeoID, _containerObjectNode.ObjectID, this, ref partInfo, HEU_PartData.PartOutputType.VOLUME, isPartEditable);
							SetupGameObjectAndTransform(partData, ParentAsset);
							bResult = partData.ProcessVolumePart(session, ref volumeInfo);
							if (!bResult)
							{
								DestroyPart(partData);
								partData = null;
							}
						}
					}
				}
				else if (partInfo.type == HAPI_PartType.HAPI_PARTTYPE_MESH)
				{
					if (partData == null)
					{
						partData = ScriptableObject.CreateInstance<HEU_PartData>();
					}

					partData.Initialize(session, partID, GeoID, _containerObjectNode.ObjectID, this, ref partInfo, HEU_PartData.PartOutputType.MESH, isPartEditable);
					SetupGameObjectAndTransform(partData, parentAsset);
					bResult = partData.ProcessMeshPart(session, ref partInfo, parentAsset.GenerateUVs, parentAsset.GenerateTangents, (GeoType == HAPI_GeoType.HAPI_GEOTYPE_INTERMEDIATE));
					if(!bResult)
					{
						DestroyPart(partData);
						partData = null;
					}
				}
				else if (partInfo.type == HAPI_PartType.HAPI_PARTTYPE_INSTANCER)
				{
					if (partData == null)
					{
						partData = ScriptableObject.CreateInstance<HEU_PartData>();
					}

					partData.Initialize(session, partID, GeoID, _containerObjectNode.ObjectID, this, ref partInfo, HEU_PartData.PartOutputType.INSTANCER, isPartEditable);
					SetupGameObjectAndTransform(partData, parentAsset);
					// We'll process instancers after
				}
				else
				{
					Debug.LogWarningFormat("Unsupported part type {0}", partInfo.type);
				}

				if (partData != null)
				{
					// Success!
					_parts.Add(partData);

					// Set unique name for the part
					string partFullName = GeneratePartFullName(partData.PartName);
					partFullName = HEU_EditorUtility.GetUniqueNameForSibling(ParentAsset.RootGameObject.transform, partFullName);
					partData.SetGameObjectName(partFullName);

					// For intermediate or default-type editable nodes, setup the HEU_AttributeStore
					if(isPartEditable)
					{
						// Commenting out for alpha build.
						//partData.SyncAttributesStore(session, _geoInfo.nodeId, ref partInfo);
					}
					else
					{
						// Remove attributes store if it has it
						partData.DestroyAttributesStore();
					}

					partData.CalculateVisibility(IsVisible());

					partData.AssignUnityTag(session);
				}
			}

#if HEU_PROFILER_ON
			Debug.LogFormat("PART PROCESS TIME:: NAME={0}, TIME={1}", HEU_SessionManager.GetString(partInfo.nameSH, session), (Time.realtimeSinceStartup - processPartStartTime));
#endif
		}

		private void SetupGameObjectAndTransform(HEU_PartData partData, HEU_HoudiniAsset parentAsset)
		{
			// Set a valid gameobject for this part
			if(partData._gameObject == null)
			{
				partData._gameObject = new GameObject();
			}

			// The parent is either the asset root, OR if this is instanced and not visible, then the HDA data is the parent
			// The parent transform is either the asset root (for a display node),
			// or the HDA_Data gameobject (for instanced, not visible, intermediate, editable non-display nodes)
			Transform partTransform = partData._gameObject.transform;
			if (partData.IsPartInstanced() 
				|| (_containerObjectNode.IsInstanced() && !_containerObjectNode.IsVisible())
				|| partData.IsPartCurve()
				|| IsIntermediateOrEditable())
			{
				partTransform.parent = parentAsset.OwnerGameObject.transform;
			}
			else
			{
				partTransform.parent = parentAsset.RootGameObject.transform;
			}

			partData._gameObject.isStatic = partTransform.parent.gameObject.isStatic;

			// Reset to origin
			partTransform.localPosition = Vector3.zero;
			partTransform.localRotation = Quaternion.identity;
			partTransform.localScale = Vector3.one;
		}

		/// <summary>
		/// Generate the instances of instancer parts.
		/// </summary>
		private void ProcessInstancerParts(HEU_SessionBase session)
		{
			int numParts = _parts.Count;
			for (int i = 0; i < numParts; ++i)
			{
				if (_parts[i].IsPartInstancer() && !_parts[i].HaveInstancesBeenGenerated())
				{
					_parts[i].GeneratePartInstances(session);
				}
			}
		}

		private void ProcessGeoCurve(HEU_SessionBase session)
		{
			HEU_HoudiniAsset parentAsset = ParentAsset;

			//bool isGeoCurveType = (_geoInfo.isEditable && _geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_CURVE);

			string curveName = GenerateGeoCurveName();
			curveName = HEU_EditorUtility.GetUniqueNameForSibling(ParentAsset.RootGameObject.transform, curveName);

			if (_geoCurve == null)
			{
				// New geo curve
				_geoCurve = HEU_Curve.CreateSetupCurve(parentAsset, Editable, curveName, GeoID, true);
			}
			else
			{
				_geoCurve.UploadParameterPreset(session, GeoID, parentAsset);
			}

			SetupGeoCurveGameObjectAndTransform(_geoCurve);
			_geoCurve.SetCurveName(curveName);

			_geoCurve.SyncFromParameters(session, parentAsset);

			bool bResult = _geoCurve.UpdateCurve(session, 0);
			if (bResult)
			{
				_geoCurve.GenerateMesh(_geoCurve._targetGameObject);
			}

			bool bIsVisible = IsVisible() && HEU_PluginSettings.Curves_ShowInSceneView;
			_geoCurve.SetCurveGeometryVisibility(bIsVisible);
		}

		private void SetupGeoCurveGameObjectAndTransform(HEU_Curve curve)
		{
			if (curve._targetGameObject == null)
			{
				curve._targetGameObject = new GameObject();
			}

			// For geo curve, the parent is the HDA_Data
			Transform curveTransform = curve._targetGameObject.transform;
			curveTransform.parent = ParentAsset.OwnerGameObject.transform;

			curve._targetGameObject.isStatic = curveTransform.parent.gameObject.isStatic;

			// Reset to origin
			curveTransform.localPosition = Vector3.zero;
			curveTransform.localRotation = Quaternion.identity;
			curveTransform.localScale = Vector3.one;
		}

		/// <summary>
		/// Clear object instances so that they can be re-created
		/// </summary>
		public void ClearObjectInstances()
		{
			int numParts = _parts.Count;
			for (int i = 0; i < numParts; ++i)
			{
				_parts[i].ObjectInstancesBeenGenerated = false;
			}
		}

		public void SetGeoInfo(HAPI_GeoInfo geoInfo)
		{
			_geoInfo = geoInfo;
		}

		/// <summary>
		/// Returns the part's full name to set on the gamepboject
		/// </summary>
		/// <param name="partName"></param>
		/// <returns></returns>
		public string GeneratePartFullName(string partName)
		{
			return _containerObjectNode.ObjectName + "_" + GeoName + "_" + partName;
		}

		public string GenerateGeoCurveName()
		{
			return "GeoCurve";
		}

		/// <summary>
		/// Returns true if any of the geo data has changed and 
		/// therefore requires regeneration.
		/// </summary>
		/// <returns>True if changes occurred internal to geo data</returns>
		public bool HasGeoNodeChanged(HEU_SessionBase session)
		{
			// Note: _geoInfo.hasMaterialChanged has been deprecated so we're not checking that

			if (!session.GetGeoInfo(GeoID, ref _geoInfo))
			{
				return false;
			}
			else if (_geoInfo.type == HAPI_GeoType.HAPI_GEOTYPE_INPUT)
			{
				return (_inputNode != null) ? _inputNode.RequiresCook : false;
			}
			else if (_geoInfo.isTemplated && !HEU_PluginSettings.CookTemplatedGeos && !_geoInfo.isEditable)
			{
				return false;
			}
			else if (!_geoInfo.hasGeoChanged)
			{
				return false;
			}
			// Commented out to allow non-display geo to be updated
			//else if (!_geoInfo.isDisplayGeo && (_geoInfo.type != HAPI_GeoType.HAPI_GEOTYPE_CURVE && !HEU_PluginSettings.CookTemplatedGeos && _geoInfo.isTemplated))
			//{
			//	return false;
			//}

			return true;
		}

		/// <summary>
		/// Apply the given HAPI transform to all parts of this node.
		/// </summary>
		/// <param name="hapiTransform">HAPI transform to apply</param>
		public void ApplyHAPITransform(ref HAPI_Transform hapiTransform)
		{
			foreach(HEU_PartData part in _parts)
			{
				part.ApplyHAPITransform(ref hapiTransform);
			}
		}

		/// <summary>
		/// Get debug info for this geo
		/// </summary>
		public void GetDebugInfo(StringBuilder sb)
		{
			int numParts = _parts != null ? _parts.Count : 0;

			sb.AppendFormat("GeoID: {0}, Name: {1}, Type: {2}, Displayable: {3}, Editable: {4}, Parts: {5}, Parent: {6}\n", GeoID, GeoName, GeoType, Displayable, Editable, numParts, ParentAsset);

			if (_parts != null)
			{
				foreach (HEU_PartData part in _parts)
				{
					part.GetDebugInfo(sb);
				}
			}
		}

		/// <summary>
		/// Returns true if this geonode is using the given material.
		/// </summary>
		/// <param name="materialData">Material data containing the material to check</param>
		/// <returns>True if this geonode is using the given material</returns>
		public bool IsUsingMaterial(HEU_MaterialData materialData)
		{
			foreach (HEU_PartData part in _parts)
			{
				if(part.IsUsingMaterial(materialData))
				{
					return true;
				}
			}
			return false;
		}

		public void GetClonableParts(List<HEU_PartData> clonableParts)
		{
			foreach (HEU_PartData part in _parts)
			{
				part.GetClonableParts(clonableParts);
			}
		}

		/// <summary>
		/// Adds gameobjects that were output from this geo node.
		/// </summary>
		/// <param name="outputObjects">List to add to</param>
		public void GetOutputGameObjects(List<GameObject> outputObjects)
		{
			foreach (HEU_PartData part in _parts)
			{
				part.GetOutputGameObjects(outputObjects);
			}
		}

		/// <summary>
		/// Returns the HEU_PartData with the given output gameobject.
		/// </summary>
		/// <param name="outputGameObject">The output gameobject to check</param>
		/// <returns>Valid HEU_PartData or null if no match</returns>
		public HEU_PartData GetHDAPartWithGameObject(GameObject outputGameObject)
		{
			HEU_PartData foundPart = null;
			foreach (HEU_PartData part in _parts)
			{
				foundPart = part.GetHDAPartWithGameObject(outputGameObject);
				if (foundPart != null)
				{
					return foundPart;
				}
			}
			return null;
		}

		/// <summary>
		/// Returns contained part with specified partID
		/// </summary>
		/// <param name="partID">The node ID to match</param>
		/// <returns>The part with partID</returns>
		public HEU_PartData GetPartFromPartID(HAPI_NodeId partID)
		{
			int numParts = _parts.Count;
			for (int i = 0; i < numParts; ++i)
			{
				if (_parts[i].PartID == partID)
				{
					return _parts[i];
				}
			}
			return null;
		}

		public void GetCurves(List<HEU_Curve> curves, bool bEditableOnly)
		{
			if (_geoCurve != null && (!bEditableOnly || _geoCurve.IsEditable()))
			{
				curves.Add(_geoCurve);
			}

			foreach (HEU_PartData part in _parts)
			{
				HEU_Curve partCurve = part.GetCurve(bEditableOnly);
				if(partCurve != null)
				{
					curves.Add(partCurve);
				}
			}
		}

		public List<HEU_PartData> GetParts()
		{
			return _parts;
		}

		/// <summary>
		/// Calculate the visibility of this geo node and its parts, based on whether the parent is visible.
		/// </summary>
		/// <param name="bParentVisibility">True if parent is visible</param>
		public void CalculateVisiblity(bool bParentVisibility)
		{
			if (_geoCurve != null)
			{
				bool curveVisiblity = bParentVisibility && HEU_PluginSettings.Curves_ShowInSceneView;
				_geoCurve.SetCurveGeometryVisibility(curveVisiblity);
			}

			int numParts = _parts.Count;
			for (int i = 0; i < numParts; ++i)
			{
				_parts[i].CalculateVisibility(bParentVisibility);
			}
		}

		public override string ToString()
		{
			return (!string.IsNullOrEmpty(_geoName) ? ("GeoNode: " + _geoName) : base.ToString());
		}
	}

}   // HoudiniEngineUnity
						 