﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using System.Collections.Generic;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// The root object of a Houdini Engine asset.
	/// Used for organizing hierarchy, and more importantly displaying custom UI.
	/// </summary>
	[SelectionBase]
	[ExecuteInEditMode]	// Needed to get OnDestroy callback when deleted in Editor
	public class HEU_HoudiniAssetRoot : MonoBehaviour
	{
		// Reference to the actual Houdini Engine asset gamebobject which contains
		// all the data and logic to work with Houdini Engine
		public HEU_HoudiniAsset _houdiniAsset;

		public List<GameObject> _bakeTargets = new List<GameObject>();


		/// <summary>
		/// Callback when asset is deleted. Removes assset from Houdini session if in Editor.
		/// </summary>
		void OnDestroy()
		{
			// Destroy the asset from session or permanently. 
			// The following checks make sure to only delete if the scene is closing, 
			// or asset has been user deleted. 
			// Skips if its just transitioning into or out of play mode.
			// TODO RUNTIME: if/when we support runtime, should only do the Application.isEditor check if in Editor			
			if( _houdiniAsset != null && HEU_EditorUtility.IsEditorNotInPlayModeAndNotGoingToPlayMode())
			{
				// Delete mesh data if this asset hasn't been saved and it is a user invoked delete event.
				// Otherwise just remove from session.
				// TODO: for saved assets, we need to handle case where user deletes but does not save scene after
				if (!_houdiniAsset.IsAssetSavedInScene() && (Event.current != null && (Event.current.commandName.Equals("Delete") || Event.current.commandName.Equals("SoftDelete"))))
				{
					_houdiniAsset.DestroyGeneratedMeshData(bRegisterUndo: true);
				}

				_houdiniAsset.DeleteAllGeneratedData();

				// TODO: Handle Event.current.commandName.Equals("Duplicate")
			}
		}

		/// <summary>
		/// Removes all Houdini Engine data from this asset.
		/// Leaves this gameobject and its children including Unity-specific
		/// components like geometry, materials, etc.
		/// </summary>
		public void RemoveHoudiniEngineAssetData()
		{
			HEU_EditorUtility.UndoRecordObject(this, "Clear References");
			// TODO: try Undo.RegisterCompleteObjectUndo or  RegisterFullObjectHierarchyUndo

			if (_houdiniAsset != null)
			{
				// We'll do a simple DestroyImmediate.
				// No need to destroy the object, geo nodes, and parts
				// since Unity's GC will handle them.

				GameObject tempGO = _houdiniAsset.gameObject;
				_houdiniAsset = null;
				HEU_GeneralUtility.DestroyImmediate(tempGO, bRegisterUndo:true);
			}

			ClearHoudiniEngineReferences();
			DestroyRootComponent(this);

			HEU_EditorUtility.UndoCollapseCurrentGroup();
		}

		public void ClearHoudiniEngineReferences()
		{
			_houdiniAsset = null;
			_bakeTargets.Clear();
		}

		public static void DestroyRootComponent(HEU_HoudiniAssetRoot assetRoot)
		{
			HEU_GeneralUtility.DestroyImmediate(assetRoot, bRegisterUndo:true);
		}
	}


}   // HoudiniEngineUnity