﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/


using System.Collections;
using System.Collections.Generic;
using UnityEngine;


namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;


	// <summary>
	/// Represents a general node for sending data upstream to Houdini.
	/// Currently only supports sending geometry upstream.
	/// Specify input data as file (eg. bgeo), HDA, and Unity gameobjects.
	/// </summary>
	public class HEU_InputNode : ScriptableObject
	{
		// DATA -------------------------------------------------------------------------------------------------------

		// The type of input node based on how it was specified in the HDA
		public enum InputNodeType
		{
			CONNECTION,		// As an asset connection
			NODE,			// Pure input asset node
			PARAMETER,		// As an input parameter
		}

		[SerializeField]
		private InputNodeType _inputNodeType;

		public InputNodeType InputType { get { return _inputNodeType; } }

		// The type of input data set by user
		public enum InputObjectType
		{
			HDA,
			UNITY_MESH,
			//CURVE
		}

		[SerializeField]
		private InputObjectType _inputObjectType = InputObjectType.UNITY_MESH;

		[SerializeField]
		private InputObjectType _pendingInputObjectType = InputObjectType.UNITY_MESH;

		// The IDs of the object merge created for the input objects
		[SerializeField]
		private List<HEU_InputObjectInfo> _inputObjects = new List<HEU_InputObjectInfo>();

		[SerializeField]
		private List<HAPI_NodeId> _inputObjectsConnectedAssetIDs = new List<HAPI_NodeId>();

		[SerializeField]
		private GameObject _inputAsset;

		[SerializeField]
		private bool _inputAssetConnected;

		[SerializeField]
		private HAPI_NodeId _nodeID;

		[SerializeField]
		private int _inputIndex;

		[SerializeField]
		private bool _requiresCook;

		public bool RequiresCook { get { return _requiresCook; } set { _requiresCook = value; } }

		[SerializeField]
		private bool _requiresUpload;

		public bool RequiresUpload { get { return _requiresUpload; } set { _requiresUpload = value; } }

		[SerializeField]
		private string _inputName;

		public string InputName { get { return _inputName; } }

		[SerializeField]
		private string _paramName;

		public string ParamName { get { return _paramName; } set { _paramName = value; } }

		[SerializeField]
		private HAPI_NodeId _connectedNodeID = HEU_Defines.HEU_INVALID_NODE_ID;

		[SerializeField]
		private bool _keepWorldTransform;

		// If true, sets the SOP/merge (object merge) node to use INTO_THIS_OBJECT transform type. Otherwise NONE.
		public bool KeepWorldTransform { get { return _keepWorldTransform; } set { _keepWorldTransform = value; } }

		[SerializeField]
		private bool _packGeometryBeforeMerging;

		// Acts same as SOP/merge (object merge) Pack Geometry Before Merging parameter value.
		public bool PackGeometryBeforeMerging { get { return _packGeometryBeforeMerging; } set { _packGeometryBeforeMerging = value; } }

		[SerializeField]
		private HEU_HoudiniAsset _parentAsset;

		public enum InputActions
		{
			ACTION,
			DELETE,
			INSERT
		}

		// LOGIC ------------------------------------------------------------------------------------------------------

		public static HEU_InputNode CreateSetupInput(HAPI_NodeId nodeID, int inputIndex, string inputName, InputNodeType inputNodeType, HEU_HoudiniAsset parentAsset)
		{
			HEU_InputNode newInput = ScriptableObject.CreateInstance<HEU_InputNode>();
			newInput._nodeID = nodeID;
			newInput._inputIndex = inputIndex;
			newInput._inputName = inputName;
			newInput._inputNodeType = inputNodeType;
			newInput._parentAsset = parentAsset;

			newInput._requiresUpload = false;
			newInput._requiresCook = false;

			return newInput;
		}

		public void DestroyAllData(HEU_SessionBase session)
		{
			ClearUICache();

			DisconnectAndDestroyInputAssets(session);
		}

		private void ResetInputObjectTransforms()
		{
			for(int i = 0; i < _inputObjects.Count; ++i)
			{
				_inputObjects[i]._syncdTransform = Matrix4x4.identity;
			}
		}

		private HEU_InputObjectInfo CreateInputObjectInfo(GameObject inputGameObject)
		{
			HEU_InputObjectInfo newObjectInfo = new HEU_InputObjectInfo();
			newObjectInfo._gameObject = inputGameObject;

			return newObjectInfo;
		}

		public void InsertInputObject(int index, GameObject newInputGameObject)
		{
			if(index >= 0 && index < _inputObjects.Count)
			{
				_inputObjects.Insert(index, CreateInputObjectInfo(newInputGameObject));
			}
			else
			{
				Debug.LogErrorFormat("Insert index {0} out of range (number of items is {1})", index, _inputObjects.Count);
			}
		}

		public HEU_InputObjectInfo GetInputObject(int index)
		{
			if (index >= 0 && index < _inputObjects.Count)
			{
				return _inputObjects[index];
			}
			else
			{
				Debug.LogErrorFormat("Get index {0} out of range (number of items is {1})", index, _inputObjects.Count);
			}
			return null;
		}

		public void AddInputObjectAtEnd(GameObject newInputGameObject)
		{
			_inputObjects.Add(CreateInputObjectInfo(newInputGameObject));
		}

		public void RemoveInputObject(int index)
		{
			if (index >= 0 && index < _inputObjects.Count)
			{
				_inputObjects.RemoveAt(index);
			}
			else
			{
				Debug.LogErrorFormat("Remove index {0} out of range (number of items is {1})", index, _inputObjects.Count);
			}
		}

		public int NumInputObjects()
		{
			return _inputObjects.Count;
		}

		private void ChangeInputType(HEU_SessionBase session, InputObjectType newType)
		{
			if(newType == _inputObjectType)
			{
				return;
			}

			DisconnectAndDestroyInputAssets(session);

			_inputObjectType = newType;
			_pendingInputObjectType = _inputObjectType;
		}

		/// <summary>
		/// Reset the connected state so that any previous connection will be remade
		/// </summary>
		public void ResetConnectionForForceUpdate(HEU_SessionBase session)
		{
			if (_inputObjectType == InputObjectType.HDA)
			{
				if (_inputAssetConnected)
				{
					// By disconnecting here, we can then properly reconnect again.
					// This is needed when loading a saved scene and recooking.
					DisconnectInputAssetActor(session);
				}
			}
		}

		public void UploadInput(HEU_SessionBase session)
		{
			if (_nodeID == HEU_Defines.HEU_INVALID_NODE_ID)
			{
				Debug.LogErrorFormat("Input Node ID is invalid. Unable to upload input. Try recooking.");
				return;
			}

			if(_pendingInputObjectType != _inputObjectType)
			{
				ChangeInputType(session, _pendingInputObjectType);
			}

			if(_inputObjectType == InputObjectType.UNITY_MESH)
			{
				if(_inputObjects == null || _inputObjects.Count == 0)
				{
					DisconnectAndDestroyInputAssets(session);
				}
				else
				{
					DisconnectAndDestroyInputAssets(session);

					bool bResult = HEU_HAPIUtility.CreateInputNodeWithMultiObjects(session, _nodeID, ref _connectedNodeID, ref _inputObjects, ref _inputObjectsConnectedAssetIDs);
					if(!bResult)
					{
						DisconnectAndDestroyInputAssets(session);
						return;
					}

					ConnectInputNode(session);

					if(!UploadObjectMergeTransformType(session))
					{
						Debug.LogErrorFormat("Failed to upload object merge transform type!");
						return;
					}

					if (!UploadObjectMergePackGeometry(session))
					{
						Debug.LogErrorFormat("Failed to upload object merge pack geometry value!");
						return;
					}
				}
			}
			else if(_inputObjectType == InputObjectType.HDA)
			{
				// Connect HDA. Note only 1 connection supported.

				if(_inputAsset != null)
				{
					HEU_HoudiniAssetRoot inputAssetRoot = _inputAsset.GetComponent<HEU_HoudiniAssetRoot>();
					if(inputAssetRoot == null || !inputAssetRoot._houdiniAsset.IsAssetValidInHoudini(session))
					{
						Debug.LogWarningFormat("The input GameObject {0} is not a valid HDA asset.", _inputAsset.name);

						if (_inputAssetConnected)
						{
							DisconnectInputAssetActor(session);
						}
					}
					else
					{
						ConnectInputAssetActor(session);
					}
				}
				else if (_inputAssetConnected)
				{
					DisconnectInputAssetActor(session);
				}
			}
			//else if (_inputObjectType == InputObjectType.CURVE)
			//{
				// TODO INPUT NODE - create new Curve SOP (add HEU_Curve here?)
			//}
			else
			{
				Debug.LogErrorFormat("Unsupported input type {0}. Unable to upload input.", _inputObjectType);
			}

			RequiresUpload = false;
			RequiresCook = true;

			ClearUICache();
		}

		private void ConnectInputAssetActor(HEU_SessionBase session)
		{
			if(_inputAssetConnected)
			{
				return;
			}

			HEU_HoudiniAssetRoot inputAssetRoot = _inputAsset != null ? _inputAsset.GetComponent<HEU_HoudiniAssetRoot>() : null;
			if (inputAssetRoot != null && inputAssetRoot._houdiniAsset.IsAssetValidInHoudini(session))
			{
				_connectedNodeID = inputAssetRoot._houdiniAsset.AssetID;

				ConnectInputNode(session);

				_parentAsset.ConnectToUpstream(inputAssetRoot._houdiniAsset);

				_inputAssetConnected = true;
			}
		}

		private void DisconnectInputAssetActor(HEU_SessionBase session)
		{
			if (!_inputAssetConnected)
			{
				return;
			}

			if (_inputNodeType == InputNodeType.PARAMETER)
			{
				HEU_ParameterData paramData = _parentAsset.Parameters.GetParameter(_paramName);
				if(paramData == null)
				{
					Debug.LogErrorFormat("Unable to find parameter with name {0}!", _paramName);
				}
				else if (!session.SetParamStringValue(_nodeID, "", paramData.ParmID, 0))
				{
					Debug.LogErrorFormat("Unable to clear object path parameter for input node!");
				}
			}
			else if(session != null && _nodeID != HEU_Defines.HEU_INVALID_NODE_ID)
			{
				session.DisconnectNodeInput(_nodeID, _inputIndex, false);
			}

			HEU_HoudiniAssetRoot inputAssetRoot = _inputAsset != null ? _inputAsset.GetComponent<HEU_HoudiniAssetRoot>() : null;
			if (inputAssetRoot != null)
			{
				_parentAsset.DisconnectFromUpstream(inputAssetRoot._houdiniAsset);
			}

			_connectedNodeID = HEU_Defines.HEU_INVALID_NODE_ID;
			_inputAssetConnected = false;
		}

		private void ConnectInputNode(HEU_SessionBase session)
		{
			// Connect node input
			
			if (_inputNodeType == InputNodeType.PARAMETER)
			{
				if (string.IsNullOrEmpty(_paramName))
				{
					Debug.LogErrorFormat("Invalid parameter name for input node of parameter type!");
					return;
				}

				if(!session.SetParamNodeValue(_nodeID, _paramName, _connectedNodeID))
				{
					Debug.LogErrorFormat("Unable to connect to input node!");
					return;
				}
			}
			else
			{
				if(!session.ConnectNodeInput(_nodeID, _inputIndex, _connectedNodeID))
				{
					Debug.LogErrorFormat("Unable to connect to input node!");
					return;
				}
			}
		}

		private void DisconnectAndDestroyInputAssets(HEU_SessionBase session)
		{
			if (_inputObjectType == InputObjectType.HDA)
			{
				DisconnectInputAssetActor(session);

				//_connectedNodeID = HEU_Defines.HEU_INVALID_NODE_ID;

				//if (_inputNodeType == InputNodeType.PARAMETER)
				//{

				//}
			}

			if (session != null)
			{
				foreach (HAPI_NodeId nodeID in _inputObjectsConnectedAssetIDs)
				{
					session.DeleteNode(nodeID);
				}
				_inputObjectsConnectedAssetIDs.Clear();

				if (_connectedNodeID != HEU_Defines.HEU_INVALID_NODE_ID && HEU_HAPIUtility.IsAssetValidInHoudini(session, _connectedNodeID))
				{
					session.DeleteNode(_connectedNodeID);
				}
			}

			_connectedNodeID = HEU_Defines.HEU_INVALID_NODE_ID;
		}

		public bool UploadObjectMergeTransformType(HEU_SessionBase session)
		{
			if(_nodeID == HEU_Defines.HAPI_INVALID_PARM_ID)
			{
				return false;
			}

			if (_inputObjectType != InputObjectType.UNITY_MESH)
			{
				return false;
			}

			int transformType = _keepWorldTransform ? 1 : 0;

			if (_inputNodeType == InputNodeType.PARAMETER)
			{
				return session.SetParamIntValue(_nodeID, HEU_Defines.HAPI_OBJMERGE_TRANSFORM_PARAM, 0, transformType);
			}
			else
			{
				HAPI_NodeId inputNodeID = HEU_Defines.HEU_INVALID_NODE_ID;
				if (session.QueryNodeInput(_nodeID, _inputIndex, out inputNodeID, true))
				{
					return session.SetParamIntValue(inputNodeID, HEU_Defines.HAPI_OBJMERGE_TRANSFORM_PARAM, 0, transformType);
				}
			}
			return false;
		}

		private bool UploadObjectMergePackGeometry(HEU_SessionBase session)
		{
			if (_nodeID == HEU_Defines.HAPI_INVALID_PARM_ID)
			{
				return false;
			}

			if (_inputObjectType != InputObjectType.UNITY_MESH)
			{
				return false;
			}

			HAPI_NodeId inputNodeID = HEU_Defines.HEU_INVALID_NODE_ID;
			if (session.QueryNodeInput(_nodeID, _inputIndex, out inputNodeID, false))
			{
				int packEnabled = _packGeometryBeforeMerging ? 1 : 0;
				return session.SetParamIntValue(inputNodeID, HEU_Defines.HAPI_OBJMERGE_PACK_GEOMETRY, 0, packEnabled);
			}
			return true;
		}

		public bool HasInputNodeTransformChanged()
		{
			if (_inputObjectType == InputObjectType.UNITY_MESH)
			{
				for (int i = 0; i < _inputObjects.Count; ++i)
				{
					if (_inputObjects[i]._gameObject != null)
					{
						if (_inputObjects[i]._useTransformOverride)
						{
							if (!HEU_HAPIUtility.IsSameTransform(ref _inputObjects[i]._syncdTransform, ref _inputObjects[i]._translateOverride, ref _inputObjects[i]._rotateOverride, ref _inputObjects[i]._scaleOverride))
							{
								return true;
							}
						}
						else if (_inputObjects[i]._gameObject.transform.localToWorldMatrix != _inputObjects[i]._syncdTransform)
						{
							return true;
						}
					}
				}
			}

			return false;
		}

		public bool UploadInputObjectTransforms(HEU_SessionBase session)
		{
			if (_nodeID == HEU_Defines.HAPI_INVALID_PARM_ID)
			{
				return false;
			}

			if (_inputObjectType != InputObjectType.UNITY_MESH)
			{
				return false;
			}

			for (int i = 0; i < _inputObjects.Count; ++i)
			{
				if(_inputObjects[i]._gameObject == null)
				{
					continue;
				}

				HEU_HAPIUtility.UploadInputObjectTransform(session, _inputObjects[i], _inputObjectsConnectedAssetIDs[i]);
			}

			return false;
		}

		/// <summary>
		/// Force cook upstream connected asset if its not valid in given session.
		/// </summary>
		/// <param name="session"></param>
		public void CookUpstreamConnectedAsset(HEU_SessionBase session)
		{
			if(_inputObjectType == InputObjectType.HDA && _inputAssetConnected && _inputAsset != null)
			{
				HEU_HoudiniAssetRoot inputAssetRoot = _inputAsset.GetComponent<HEU_HoudiniAssetRoot>();
				if (inputAssetRoot != null && !inputAssetRoot._houdiniAsset.IsAssetValidInHoudini(session))
				{
					inputAssetRoot._houdiniAsset.RequestCook(false, false, true);
				}
			}
		}

		// UI CACHE ---------------------------------------------------------------------------------------------------

		public HEU_InputNodeUICache _uiCache;

		public void ClearUICache()
		{
			_uiCache = null;
		}
	}

	// Container for each input object in this node
	[System.Serializable]
	public class HEU_InputObjectInfo
	{
		// Gameobject containing mesh
		public GameObject _gameObject;

		// The last upload transform, for diff checks
		public Matrix4x4 _syncdTransform = Matrix4x4.identity;

		// Whether to use the transform override instead of GameObject's transform
		public bool _useTransformOverride = false;

		// Transform override
		public Vector3 _translateOverride = Vector3.zero;
		public Vector3 _rotateOverride = Vector3.zero;
		public Vector3 _scaleOverride = Vector3.one;
	}

	// UI cache container
	public class HEU_InputNodeUICache
	{
#if UNITY_EDITOR
		public UnityEditor.SerializedObject _inputNodeSerializedObject;

		public UnityEditor.SerializedProperty _inputObjectTypeProperty;

		public UnityEditor.SerializedProperty _keepWorldTransformProperty;
		public UnityEditor.SerializedProperty _packBeforeMergeProperty;

		public UnityEditor.SerializedProperty _inputObjectsProperty;

		public UnityEditor.SerializedProperty _inputAssetProperty;
#endif

		public class HEU_InputObjectUICache
		{
#if UNITY_EDITOR
			public UnityEditor.SerializedProperty _gameObjectProperty;
			public UnityEditor.SerializedProperty _transformOverrideProperty;
			public UnityEditor.SerializedProperty _translateProperty;
			public UnityEditor.SerializedProperty _rotateProperty;
			public UnityEditor.SerializedProperty _scaleProperty;
#endif
		}

		public List<HEU_InputObjectUICache> _inputObjectCache = new List<HEU_InputObjectUICache>();
	}

}   // HoudiniEngineUnity