﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

// Uncomment to profile
//#define HEU_PROFILER_ON

using System;
using System.Text;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;


namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;
	using HAPI_ParmId = System.Int32;
	using HAPI_StringHandle = System.Int32;


	/// <summary>
	/// Represents a Part object containing mesh / geometry/ attribute data.
	/// </summary>
	public class HEU_PartData : ScriptableObject
	{
		//	DATA ------------------------------------------------------------------------------------------------------

		[SerializeField]
		private HAPI_PartId _partID;
		public HAPI_PartId PartID { get { return _partID; } }

		[SerializeField]
		private string _partName;
		public string PartName { get { return _partName; } }

		[SerializeField]
		private HAPI_NodeId _objectNodeID;

		[SerializeField]
		private HAPI_NodeId _geoID;

		[SerializeField]
		private HAPI_PartType _partType;

		[SerializeField]
		public GameObject _gameObject;

		[SerializeField]
		private HEU_GeoNode _geoNode;

		public HEU_GeoNode ParentGeoNode { get { return _geoNode; } }

		public HEU_HoudiniAsset ParentAsset { get { return (_geoNode != null) ? _geoNode.ParentAsset : null; } }

		public bool IsPartInstancer() { return _partType == HAPI_PartType.HAPI_PARTTYPE_INSTANCER; }

		[SerializeField]
		private bool _isPartInstanced;

		public bool IsPartInstanced() { return _isPartInstanced; }

		[SerializeField]
		private int _partPointCount;

		public int GetPartPointCount() { return _partPointCount; }

		[SerializeField]
		private bool _isObjectInstancer;

		public bool IsObjectInstancer() { return _isObjectInstancer; }

		[SerializeField]
		private bool _objectInstancesGenerated;

		public bool ObjectInstancesBeenGenerated { get { return _objectInstancesGenerated; } set { _objectInstancesGenerated = value; } }

		[SerializeField]
		private List<HEU_ObjectInstanceInfo> _objectInstanceInfos;

		// Store volume position to use when applying transform
		[SerializeField]
		private Vector3 _volumePosition;

		[SerializeField]
		private UnityEngine.Object _assetDBTerrainData;

		public bool IsPartVolume() { return _partOutputType == PartOutputType.VOLUME; }

		public bool IsPartCurve() { return _partOutputType == PartOutputType.CURVE; }

		[SerializeField]
		private bool _isPartEditable;

		public bool IsPartEditable() { return _isPartEditable; }

		public enum PartOutputType
		{
			NONE,
			MESH,
			VOLUME,
			CURVE,
			INSTANCER
		}
		[SerializeField]
		private PartOutputType _partOutputType;

		[SerializeField]
		private HEU_Curve _curve;

		[SerializeField]
		private HEU_AttributesStore _attributesStore;

		[SerializeField]
		private bool _haveInstancesBeenGenerated;

		public bool HaveInstancesBeenGenerated() { return _haveInstancesBeenGenerated; }


		//  LOGIC -----------------------------------------------------------------------------------------------------


		public HEU_PartData()
		{
			_partID = HEU_Defines.HEU_INVALID_NODE_ID;
			_geoID = HEU_Defines.HEU_INVALID_NODE_ID;
			_objectNodeID = HEU_Defines.HEU_INVALID_NODE_ID;
			_partOutputType = PartOutputType.NONE;
		}

		public void Initialize(HEU_SessionBase session, HAPI_PartId partID, HAPI_NodeId geoID, HAPI_NodeId objectNodeID, HEU_GeoNode geoNode, ref HAPI_PartInfo partInfo, HEU_PartData.PartOutputType partOutputType, bool isEditable)
		{
			_partID = partID;
			_geoID = geoID;
			_objectNodeID = objectNodeID;
			_geoNode = geoNode;

			_partOutputType = partOutputType;
			_partType = partInfo.type;
			_partName = HEU_SessionManager.GetString(partInfo.nameSH, session);
			_isPartInstanced = partInfo.isInstanced;
			_partPointCount = partInfo.pointCount;
			_isPartEditable = isEditable;

			_objectInstancesGenerated = false;
			_objectInstanceInfos = new List<HEU_ObjectInstanceInfo>();

			//Debug.Log("PartData initialized with ID: " + partID);
		}

		public void SetGameObjectName(string name)
		{
			_gameObject.name = name;
		}

		/// <summary>
		/// Destroy generated data such as instances and gameobject.
		/// </summary>
		public void DestroyAllData()
		{
			ClearObjectInstanceInfos();

			if(_curve != null)
			{
				if (ParentAsset != null)
				{
					ParentAsset.RemoveCurve(_curve);
				}
				_curve.DestroyAllData();
				HEU_GeneralUtility.DestroyImmediate(_curve);
				_curve = null;
			}

			if(_attributesStore != null)
			{
				DestroyAttributesStore();
			}

			if (_gameObject != null)
			{
				HEU_HAPIUtility.DestroyGameObject(_gameObject);
				_gameObject = null;
			}
		}

		/// <summary>
		/// Destroy the generated mesh data.
		/// </summary>
		/// <param name="bRegisterUndo">Register Undo action</param>
		public void DestroyGeneratedMeshData(bool bRegisterUndo)
		{
			if (_gameObject != null)
			{
				HEU_HAPIUtility.DestroyGameObjectMeshData(_gameObject, bRegisterUndo: bRegisterUndo);
			}
		}

		private static bool GenerateMeshUsingGeoCache(HEU_SessionBase session, HEU_HoudiniAsset asset, GameObject gameObject, 
			HEU_GenerateGeoCache geoCache, bool bGenerateUVs, bool bGenerateTangents, bool bGeoIntermediate, bool bPartInstanced)
		{
#if HEU_PROFILER_ON
			float generateMeshTime = Time.realtimeSinceStartup;
#endif

			string collisionGroupName = HEU_PluginSettings.CollisionGroupName;
			string renderCollisionGroupName = HEU_PluginSettings.RenderedCollisionGroupName;

			// Stores submesh data based on material key (ie. a submesh for each unique material)

			// Unity requires that if using multiple materials in the same GameObject, then we
			// need to create corresponding number of submeshes as materials.
			// So we'll create a submesh for each material in use. 
			// Each submesh will have a list of vertices and their attributes which
			// we'll collect in a helper class (HEU_MeshData).
			// Once we collected all the submesh data, we create a CombineInstance for each
			// submesh, then combine it while perserving the submeshes.
			Dictionary<int, HEU_MeshData> subMeshesMap = new Dictionary<int, HEU_MeshData>();

			string defaultMaterialName = HEU_HoudiniAsset.GenerateDefaultMaterialName(geoCache.GeoID, geoCache.PartID);
			int defaultMaterialKey = HEU_MaterialFactory.MaterialNameToKey(defaultMaterialName);

			int singleFaceUnityMaterialKey = HEU_Defines.HEU_INVALID_MATERIAL;
			int singleFaceHoudiniMaterialKey = HEU_Defines.HEU_INVALID_MATERIAL;

			// Now go through each group data and acquire the vertex data.
			// We'll create the collider mesh rightaway and assign to the gameobject.
			int numCollisionMeshes = 0;
			foreach (KeyValuePair<string, int[]> groupSplitFacesPair in geoCache._groupSplitVertexIndices)
			{
				string groupName = groupSplitFacesPair.Key;
				int[] groupVertexList = groupSplitFacesPair.Value;

				bool bIsCollidable = groupName.Contains(collisionGroupName);
				bool bIsRenderCollidable = groupName.Contains(renderCollisionGroupName);
				if (bIsCollidable || bIsRenderCollidable)
				{
					if (numCollisionMeshes > 0)
					{
						Debug.LogWarningFormat("More than 1 collision mesh detected for part {0}.\nOnly a single collision mesh is supported per part.", geoCache._partName);
					}

					if (geoCache._partInfo.type == HAPI_PartType.HAPI_PARTTYPE_BOX)
					{
						// Box collider

						HAPI_BoxInfo boxInfo = new HAPI_BoxInfo();
						if (session.GetBoxInfo(geoCache.GeoID, geoCache.PartID, ref boxInfo))
						{
							BoxCollider boxCollider = HEU_GeneralUtility.GetOrCreateComponent<BoxCollider>(gameObject);

							boxCollider.center = new Vector3(-boxInfo.center[0], boxInfo.center[1], boxInfo.center[2]);
							boxCollider.size = new Vector3(boxInfo.size[0] * 2f, boxInfo.size[1] * 2f, boxInfo.size[2] * 2f);
							// TODO: Should we apply the box info rotation here to the box collider?
							//		 If so, it should be in its own gameobject?
						}
					}
					else if (geoCache._partInfo.type == HAPI_PartType.HAPI_PARTTYPE_SPHERE)
					{
						// Sphere collider

						HAPI_SphereInfo sphereInfo = new HAPI_SphereInfo();
						if (session.GetSphereInfo(geoCache.GeoID, geoCache.PartID, ref sphereInfo))
						{
							SphereCollider sphereCollider = HEU_GeneralUtility.GetOrCreateComponent<SphereCollider>(gameObject);

							sphereCollider.center = new Vector3(-sphereInfo.center[0], sphereInfo.center[1], sphereInfo.center[2]);
							sphereCollider.radius = sphereInfo.radius;
						}
					}
					else
					{
						// Mesh collider

						List<Vector3> collisionVertices = new List<Vector3>();
						for (int v = 0; v < groupVertexList.Length; ++v)
						{
							int index = groupVertexList[v];
							if (index >= 0 && index < geoCache._posAttr.Length)
							{
								collisionVertices.Add(new Vector3(-geoCache._posAttr[index * 3], geoCache._posAttr[index * 3 + 1], geoCache._posAttr[index * 3 + 2]));
							}
						}

						int[] collisionIndices = new int[collisionVertices.Count];
						for (int i = 0; i < collisionIndices.Length; ++i)
						{
							collisionIndices[i] = i;
						}

						Mesh collisionMesh = new Mesh();
#if UNITY_2017_3_OR_NEWER
						collisionMesh.indexFormat = UnityEngine.Rendering.IndexFormat.UInt32;
#endif
						collisionMesh.name = groupName;
						collisionMesh.vertices = collisionVertices.ToArray();
						collisionMesh.triangles = collisionIndices;
						collisionMesh.RecalculateBounds();

						MeshCollider meshCollider = HEU_GeneralUtility.GetOrCreateComponent<MeshCollider>(gameObject);
						meshCollider.sharedMesh = collisionMesh;
					}

					numCollisionMeshes++;
				}


				if (bIsCollidable && !bIsRenderCollidable)
				{
					continue;
				}

				// After this point, we'll be only processing renderable geometry

				// Transfer indices for each attribute from the single large list into group lists

				float[] groupColorAttr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._colorAttrInfo, geoCache._colorAttr, ref groupColorAttr);

				float[] groupAlphaAttr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._alphaAttrInfo, geoCache._alphaAttr, ref groupAlphaAttr);

				float[] groupNormalAttr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._normalAttrInfo, geoCache._normalAttr, ref groupNormalAttr);

				float[] groupTangentsAttr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._tangentAttrInfo, geoCache._tangentAttr, ref groupTangentsAttr);

				float[] groupUVAttr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._uvAttrInfo, geoCache._uvAttr, ref groupUVAttr);

				float[] groupUV2Attr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._uv2AttrInfo, geoCache._uv2Attr, ref groupUV2Attr);

				float[] groupUV3Attr = new float[0];
				HEU_GenerateGeoCache.TransferRegularPointAttributesToVertices(groupVertexList, ref geoCache._uv3AttrInfo, geoCache._uv3Attr, ref groupUV3Attr);

				// Unity mesh creation requires # of vertices must equal # of attributes (color, normal, uvs).
				// HAPI gives us point indices. Since our attributes are via vertex, we need to therefore
				// create new indices of vertices that correspond to our attributes.

				// To reindex, we go through each index, add each attribute corresponding to that index to respective lists.
				// Then we set the index of where we added those attributes as the new index.

				int numIndices = groupVertexList.Length;
				for (int vertexIndex = 0; vertexIndex < numIndices; vertexIndex += 3)
				{
					// groupVertexList contains -1 for unused indices, and > 0 for used
					if (groupVertexList[vertexIndex] == -1)
					{
						continue;
					}

					int faceIndex = vertexIndex / 3;
					int faceMaterialID = geoCache._houdiniMaterialIDs[faceIndex];

					// Get the submesh ID for this face. Depends on whether it is a Houdini or Unity material.
					// Using default material as failsafe
					int submeshID = HEU_Defines.HEU_INVALID_MATERIAL;

					if (geoCache._unityMaterialAttrInfo.exists)
					{
						// This face might have a Unity or Substance material attribute. 
						// Formulate the submesh ID by combining the material attributes.

						if (geoCache._singleFaceUnityMaterial)
						{
							if(singleFaceUnityMaterialKey == HEU_Defines.HEU_INVALID_MATERIAL && geoCache._unityMaterialInfos.Count > 0)
							{
								// Use first material
								var unityMaterialMapEnumerator = geoCache._unityMaterialInfos.GetEnumerator();
								if(unityMaterialMapEnumerator.MoveNext())
								{
									singleFaceUnityMaterialKey = unityMaterialMapEnumerator.Current.Key;
								}
							}
							submeshID = singleFaceUnityMaterialKey;
						}
						else
						{
							int attrIndex = faceIndex;
							if (geoCache._unityMaterialAttrInfo.owner == HAPI_AttributeOwner.HAPI_ATTROWNER_PRIM || geoCache._unityMaterialAttrInfo.owner == HAPI_AttributeOwner.HAPI_ATTROWNER_POINT)
							{
								if (geoCache._unityMaterialAttrInfo.owner == HAPI_AttributeOwner.HAPI_ATTROWNER_POINT)
								{
									attrIndex = groupVertexList[vertexIndex];
								}

								string unityMaterialName = "";
								string substanceName = "";
								int substanceIndex = -1;
								submeshID = HEU_GenerateGeoCache.GetMaterialKeyFromAttributeIndex(geoCache, attrIndex, out unityMaterialName, out substanceName, out substanceIndex);
							}
							else
							{
								// (geoCache._unityMaterialAttrInfo.owner == HAPI_AttributeOwner.HAPI_ATTROWNER_DETAIL) should have been handled as geoCache._singleFaceMaterial above

								Debug.LogErrorFormat("Unity material attribute not supported for attribute type {0}!", geoCache._unityMaterialAttrInfo.owner);
							}
						}
					}

					if (submeshID == HEU_Defines.HEU_INVALID_MATERIAL)
					{
						// Check if has Houdini material assignment

						if (geoCache._houdiniMaterialIDs.Length > 0)
						{
							if (geoCache._singleFaceHoudiniMaterial)
							{
								if(singleFaceHoudiniMaterialKey == HEU_Defines.HEU_INVALID_MATERIAL)
								{
									singleFaceHoudiniMaterialKey = geoCache._houdiniMaterialIDs[0];
								}
								submeshID = singleFaceHoudiniMaterialKey;
							}
							else if(faceMaterialID > 0)
							{
								submeshID = faceMaterialID;
							}
						}

						if(submeshID == HEU_Defines.HEU_INVALID_MATERIAL)
						{
							// Use default material
							submeshID = defaultMaterialKey;
						}
					}

					if (!subMeshesMap.ContainsKey(submeshID))
					{
						// New submesh
						subMeshesMap.Add(submeshID, new HEU_MeshData());
					}

					HEU_MeshData subMeshData = subMeshesMap[submeshID];

					for (int triIndex = 0; triIndex < 3; ++triIndex)
					{
						int vertexTriIndex = vertexIndex + triIndex;
						int positionIndex = groupVertexList[vertexTriIndex];

						// Position
						Vector3 position = new Vector3(-geoCache._posAttr[positionIndex * 3 + 0], geoCache._posAttr[positionIndex * 3 + 1], geoCache._posAttr[positionIndex * 3 + 2]);
						subMeshData._vertices.Add(position);

						// Color
						if (geoCache._colorAttrInfo.exists)
						{
							Color tempColor = new Color();
							tempColor.r = Mathf.Clamp01(groupColorAttr[vertexTriIndex * geoCache._colorAttrInfo.tupleSize + 0]);
							tempColor.g = Mathf.Clamp01(groupColorAttr[vertexTriIndex * geoCache._colorAttrInfo.tupleSize + 1]);
							tempColor.b = Mathf.Clamp01(groupColorAttr[vertexTriIndex * geoCache._colorAttrInfo.tupleSize + 2]);

							if (geoCache._alphaAttrInfo.exists)
							{
								tempColor.a = Mathf.Clamp01(groupAlphaAttr[vertexTriIndex]);
							}
							else if (geoCache._colorAttrInfo.tupleSize == 4)
							{
								tempColor.a = Mathf.Clamp01(groupColorAttr[vertexTriIndex * geoCache._colorAttrInfo.tupleSize + 3]);
							}
							else
							{
								tempColor.a = 1f;
							}
							subMeshData._colors.Add(tempColor);
						}
						else
						{
							subMeshData._colors.Add(Color.white);
						}

						// Normal
						if (vertexTriIndex < groupNormalAttr.Length)
						{
							// Flip the x
							Vector3 normal = new Vector3(-groupNormalAttr[vertexTriIndex * 3 + 0], groupNormalAttr[vertexTriIndex * 3 + 1], groupNormalAttr[vertexTriIndex * 3 + 2]);
							subMeshData._normals.Add(normal);
						}
						else
						{
							// We'll be calculating normals later
							subMeshData._normals.Add(Vector3.zero);
						}

						// UV1
						if (vertexTriIndex < groupUVAttr.Length)
						{
							Vector2 uv = new Vector2(groupUVAttr[vertexTriIndex * 2 + 0], groupUVAttr[vertexTriIndex * 2 + 1]);
							subMeshData._UVs.Add(uv);
						}

						// UV2
						if (vertexTriIndex < groupUV2Attr.Length)
						{
							Vector2 uv = new Vector2(groupUV2Attr[vertexTriIndex * 2 + 0], groupUV2Attr[vertexTriIndex * 2 + 1]);
							subMeshData._UV2s.Add(uv);
						}

						// UV3
						if (vertexTriIndex < groupUV3Attr.Length)
						{
							Vector2 uv = new Vector2(groupUV3Attr[vertexTriIndex * 2 + 0], groupUV3Attr[vertexTriIndex * 2 + 1]);
							subMeshData._UV3s.Add(uv);
						}

						// Tangents
						if (bGenerateTangents && vertexTriIndex < groupTangentsAttr.Length)
						{
							// Flip the x
							Vector4 tangent = new Vector4(-groupTangentsAttr[vertexTriIndex * 4 + 0], groupTangentsAttr[vertexTriIndex * 4 + 1], groupTangentsAttr[vertexTriIndex * 4 + 2], groupTangentsAttr[vertexTriIndex * 4 + 3]);
							subMeshData._tangents.Add(tangent);
						}

						subMeshData._indices.Add(subMeshData._vertices.Count - 1);
						//Debug.LogFormat("Submesh index mat {0} count {1}", faceMaterialID, subMeshData._indices.Count);
					}

					if (!geoCache._normalAttrInfo.exists)
					{
						// To generate normals after all the submeshes have been defined, we
						// calculate and store each triangle normal, along with the list
						// of connected vertices for each vertex

						int triIndex = subMeshData._indices.Count - 3;
						int i1 = subMeshData._indices[triIndex + 0];
						int i2 = subMeshData._indices[triIndex + 1];
						int i3 = subMeshData._indices[triIndex + 2];

						// Triangle normal
						Vector3 p1 = subMeshData._vertices[i2] - subMeshData._vertices[i1];
						Vector3 p2 = subMeshData._vertices[i3] - subMeshData._vertices[i1];
						Vector3 normal = Vector3.Cross(p1, p2).normalized;
						subMeshData._triangleNormals.Add(normal);
						int normalIndex = subMeshData._triangleNormals.Count - 1;

						// Connected vertices
						geoCache._sharedNormalIndices[groupVertexList[vertexIndex + 0]].Add(new VertexEntry(submeshID, i1, normalIndex));
						geoCache._sharedNormalIndices[groupVertexList[vertexIndex + 1]].Add(new VertexEntry(submeshID, i2, normalIndex));
						geoCache._sharedNormalIndices[groupVertexList[vertexIndex + 2]].Add(new VertexEntry(submeshID, i3, normalIndex));
					}
				}
			}

			int numSubmeshes = subMeshesMap.Keys.Count;

			bool bGenerated = false;
			if (numSubmeshes > 0)
			{
				if (!geoCache._normalAttrInfo.exists)
				{
					// Normal calculation
					// Go throuch each vertex for the entire geometry and calculate the normal vector based on connected
					// vertices. This includes vertex connections between submeshes so we should get smooth transitions across submeshes.

					int numSharedNormals = geoCache._sharedNormalIndices.Length;
					for (int a = 0; a < numSharedNormals; ++a)
					{
						for (int b = 0; b < geoCache._sharedNormalIndices[a].Count; ++b)
						{
							Vector3 sumNormal = new Vector3();
							VertexEntry leftEntry = geoCache._sharedNormalIndices[a][b];
							HEU_MeshData leftSubMesh = subMeshesMap[leftEntry._meshKey];

							List<VertexEntry> rightList = geoCache._sharedNormalIndices[a];
							for (int c = 0; c < rightList.Count; ++c)
							{
								VertexEntry rightEntry = rightList[c];
								HEU_MeshData rightSubMesh = subMeshesMap[rightEntry._meshKey];

								if (leftEntry._vertexIndex == rightEntry._vertexIndex)
								{
									sumNormal += rightSubMesh._triangleNormals[rightEntry._normalIndex];
								}
								else
								{
									float dot = Vector3.Dot(leftSubMesh._triangleNormals[leftEntry._normalIndex],
										rightSubMesh._triangleNormals[rightEntry._normalIndex]);
									if (dot >= geoCache._cosineThreshold)
									{
										sumNormal += rightSubMesh._triangleNormals[rightEntry._normalIndex];
									}
								}
							}

							leftSubMesh._normals[leftEntry._vertexIndex] = sumNormal.normalized;
						}
					}
				}


				// Go through each valid submesh data and upload into a CombineInstance for combining.
				// Each CombineInstance represents a submesh in the final mesh.
				// And each submesh in that final mesh corresponds to a material.

				// Filter out only the submeshes with valid geometry
				List<Material> validMaterials = new List<Material>();
				List<int> validSubmeshes = new List<int>();

				Dictionary<int, HEU_MaterialData> assetMaterialMap = asset.GetMaterialDataMap();

				foreach (KeyValuePair<int, HEU_MeshData> meshPair in subMeshesMap)
				{
					HEU_MeshData meshData = meshPair.Value;
					if (meshData._indices.Count > 0)
					{
						int materialKey = meshPair.Key;

						// Find the material or create it
						HEU_MaterialData materialData = null;

						HEU_UnityMaterialInfo unityMaterialInfo = null;
						if (geoCache._unityMaterialInfos.TryGetValue(materialKey, out unityMaterialInfo))
						{	
							if (!assetMaterialMap.TryGetValue(materialKey, out materialData))
							{
								// Create the material
								materialData = asset.CreateUnitySubstanceMaterialData(materialKey, unityMaterialInfo._unityMaterialPath, unityMaterialInfo._substancePath, unityMaterialInfo._substanceIndex);
								assetMaterialMap.Add(materialData._materialKey, materialData);
							}
						}
						else if(!assetMaterialMap.TryGetValue(materialKey, out materialData))
						{
							if(materialKey == defaultMaterialKey)
							{
								materialData = asset.GetOrCreateDefaultMaterialInCache(session, geoCache.GeoID, geoCache.PartID, false);
							}
							else
							{
								materialData = asset.CreateHoudiniMaterialData(session, materialKey, geoCache.GeoID, geoCache.PartID);
							}
						}

						if (materialData != null)
						{
							validSubmeshes.Add(meshPair.Key);
							validMaterials.Add(materialData._material);

							if (materialData != null && bPartInstanced)
							{
								// Handle GPU instancing on material for instanced meshes

								if (materialData._materialSource == HEU_MaterialData.Source.UNITY || materialData._materialSource == HEU_MaterialData.Source.SUBSTANCE)
								{
									// For materials already existing in Unity, we simply check if GPU instancing is enabled and warn user if not
									if(!HEU_MaterialFactory.MaterialHasGPUInstancingEnabled(materialData._material))
									{
										Debug.LogWarningFormat("Material {0} used by packed primitive does not have GPU instancing enabled!", materialData._material.name);
									}
								}
								else
								{
									// Always enable GPU instancing for material generated from Houdini
									HEU_MaterialFactory.EnableGPUInstancing(materialData._material);
								}
							}
						}
					}
				}

				int validNumSubmeshes = validSubmeshes.Count;
				CombineInstance[] meshCombiner = new CombineInstance[validNumSubmeshes];
				for (int submeshIndex = 0; submeshIndex < validNumSubmeshes; ++submeshIndex)
				{
					HEU_MeshData submesh = subMeshesMap[validSubmeshes[submeshIndex]];

					CombineInstance combine = new CombineInstance();
					combine.mesh = new Mesh();
#if UNITY_2017_3_OR_NEWER
					combine.mesh.indexFormat = UnityEngine.Rendering.IndexFormat.UInt32;
#endif

					combine.mesh.SetVertices(submesh._vertices);

					combine.mesh.SetIndices(submesh._indices.ToArray(), MeshTopology.Triangles, 0);

					if (submesh._colors.Count > 0)
					{
						combine.mesh.SetColors(submesh._colors);
					}

					if (submesh._normals.Count > 0)
					{
						combine.mesh.SetNormals(submesh._normals);
					}

					if (submesh._tangents.Count > 0)
					{
						combine.mesh.SetTangents(submesh._tangents);
					}

					if (bGenerateUVs)
					{
						// TODO: revisit to test this out
						Vector2[] generatedUVs = HEU_GeometryUtility.GeneratePerTriangle(combine.mesh);
						if (generatedUVs != null)
						{
							combine.mesh.uv = generatedUVs;
						}
					}
					else if (submesh._UVs.Count > 0)
					{
						combine.mesh.SetUVs(0, submesh._UVs);
					}

					if (submesh._UV2s.Count > 0)
					{
						combine.mesh.SetUVs(1, submesh._UV2s);
					}

					if (submesh._UV3s.Count > 0)
					{
						combine.mesh.SetUVs(2, submesh._UV3s);
					}

					combine.transform = Matrix4x4.identity;
					combine.mesh.RecalculateBounds();

					//Debug.LogFormat("Number of submeshes {0}", combine.mesh.subMeshCount);

					meshCombiner[submeshIndex] = combine;
				}

				// Geometry data
				MeshFilter meshFilter = HEU_GeneralUtility.GetOrCreateComponent<MeshFilter>(gameObject);
				meshFilter.sharedMesh = new Mesh();
#if UNITY_2017_3_OR_NEWER
				meshFilter.sharedMesh.indexFormat = UnityEngine.Rendering.IndexFormat.UInt32;		
#endif
				meshFilter.sharedMesh.name = geoCache._partName + "_mesh";
				meshFilter.sharedMesh.CombineMeshes(meshCombiner, false, false);

				meshFilter.sharedMesh.RecalculateBounds();

				if (!geoCache._tangentAttrInfo.exists && bGenerateTangents)
				{
					HEU_GeometryUtility.CalculateMeshTangents(meshFilter.sharedMesh);
				}

				meshFilter.sharedMesh.UploadMeshData(true);

				// Render data
				MeshRenderer meshRenderer = HEU_GeneralUtility.GetOrCreateComponent<MeshRenderer>(gameObject);
				meshRenderer.sharedMaterials = validMaterials.ToArray();

				bGenerated = true;
			}

#if HEU_PROFILER_ON
			Debug.LogFormat("GENERATE MESH TIME:: {0}", (Time.realtimeSinceStartup - generateMeshTime));
#endif

			return bGenerated;
		}


		private int TransferRegularPointAttributesToVertices(int[] vertexList, ref HAPI_AttributeInfo attribInfo, float[] inData, ref float[] outData)
		{
			int validWedgeCount = 0;
			if (attribInfo.exists && attribInfo.tupleSize > 0)
			{
				int wedgeCount = vertexList.Length;

				// Re-indexed wedges
				outData = new float[wedgeCount * attribInfo.tupleSize];

				int lastValidWedgeIndex = 0;
				for (int wedgeIndex = 0; wedgeIndex < wedgeCount; ++wedgeIndex)
				{
					int vertexIndex = vertexList[wedgeIndex];
					if (vertexIndex == -1)
					{
						continue;
					}

					validWedgeCount++;

					int primIndex = wedgeIndex / 3;
					int saveIndex = 0;
					float value = 0;

					for (int attribIndex = 0; attribIndex < attribInfo.tupleSize; ++attribIndex)
					{
						switch (attribInfo.owner)
						{
							case HAPI_AttributeOwner.HAPI_ATTROWNER_POINT:
							{
								value = inData[vertexIndex * attribInfo.tupleSize + attribIndex];
								break;
							}
							case HAPI_AttributeOwner.HAPI_ATTROWNER_PRIM:
							{
								value = inData[primIndex * attribInfo.tupleSize + attribIndex];
								break;
							}
							case HAPI_AttributeOwner.HAPI_ATTROWNER_DETAIL:
							{
								value = inData[attribIndex];
								break;
							}
							case HAPI_AttributeOwner.HAPI_ATTROWNER_VERTEX:
							{
								value = inData[wedgeIndex * attribInfo.tupleSize + attribIndex];
								break;
							}
							default:
							{
								Debug.LogAssertion("Unsupported attribute owner " + attribInfo.owner);
								continue;
							}
						}

						saveIndex = lastValidWedgeIndex * attribInfo.tupleSize + attribIndex;
						outData[saveIndex] = value;
					}

					lastValidWedgeIndex++;
				}
			}

			return validWedgeCount;
		}

		
		/// <summary>
		/// Assign the Unity tag to the GameObject if found on the part.
		/// </summary>
		public void AssignUnityTag(HEU_SessionBase session)
		{
			HAPI_AttributeInfo tagAttrInfo = new HAPI_AttributeInfo();
			int[] tagAttr = new int[0];
			HEU_GeneralUtility.GetAttribute(session, _geoID, PartID, HEU_PluginSettings.UnityTagAttributeName, ref tagAttrInfo, ref tagAttr, session.GetAttributeStringData);
			if (tagAttrInfo.exists)
			{
				string tag = HEU_SessionManager.GetString(tagAttr[0]);
				if (tag.Length > 0)
				{
					try
					{
						_gameObject.tag = tag;
					}
					catch (Exception ex)
					{
						Debug.LogWarning("Tag exception: " + ex.ToString());
						Debug.LogWarningFormat("Unity tag '{0}' does not exist for current project. Add the tag in order to use it!");
					}
				}
			}
		}

		/// <summary>
		/// Apply given HAPI transform to this part's gameobject
		/// </summary>
		/// <param name="hapiTransform">The HAPI transform to apply</param>
		public void ApplyHAPITransform(ref HAPI_Transform hapiTransform)
		{
			//Debug.Log("PartData:: ApplyHAPITransforms");
			
			//Matrix4x4 hapiTransformMatrix = HEU_HAPIUtility.GetMatrixFromHAPITransform(ref hapiTransform);
			
			if (IsPartVolume())
			{
				// For volumes, we need to also apply the volume transform due to Unity's terrain having
				// different origin.
				/*
				HEU_SessionBase session = HEU_SessionManager.GetValidCurrentSession();

				HAPI_VolumeInfo volumeInfo = new HAPI_VolumeInfo();
				bool bResult = session.GetVolumeInfo(_geoID, _partID, ref volumeInfo);
				if (bResult)
				{
					Matrix4x4 volumeTransformMatrix = HEU_HAPIUtility.GetMatrixFromHAPITransform(ref volumeInfo.transform, false);

					// Need to override the position y due to Unity not supporting negative height values.
					Vector3 position = HEU_HAPIUtility.GetPosition(ref volumeTransformMatrix);
					position.y += _terrainMinHeight;
					HEU_HAPIUtility.SetMatrixPosition(ref volumeTransformMatrix, ref position);

					hapiTransformMatrix = volumeTransformMatrix * hapiTransformMatrix;
				}
				*/

				HAPI_Transform hapiTransformVolume = hapiTransform;
				hapiTransform.position[0] += _volumePosition[0];
				hapiTransform.position[1] += _volumePosition[1];
				hapiTransform.position[2] += _volumePosition[2];

				HEU_HAPIUtility.ApplyLocalTransfromFromHoudiniToUnity(ref hapiTransformVolume, _gameObject.transform);
			}
			else
			{
				HEU_HAPIUtility.ApplyLocalTransfromFromHoudiniToUnity(ref hapiTransform, _gameObject.transform);
			}
			
			//HEU_HAPIUtility.ApplyMatrixToLocalTransform(ref hapiTransformMatrix, _gameObject.transform);
		}

		/// <summary>
		/// Get debug info for this part
		/// </summary>
		public void GetDebugInfo(StringBuilder sb)
		{
			sb.AppendFormat("PartID: {0}, PartName: {1}, ObjectID: {2}, GeoID: {3}, PartType: {4}, GameObject: {5}\n", PartID, PartName, _objectNodeID, _geoID, _partType, _gameObject);
		}

		/// <summary>
		/// Returns true if this part's mesh is using the given material.
		/// </summary>
		/// <param name="materialData">Material data containing the material to check</param>
		/// <returns>True if this part is using the given material</returns>
		public bool IsUsingMaterial(HEU_MaterialData materialData)
		{
			MeshRenderer meshRenderer = _gameObject.GetComponent<MeshRenderer>();
			if (meshRenderer != null)
			{
				Material[] inUseMaterials = meshRenderer.sharedMaterials;
				foreach (Material material in inUseMaterials)
				{
					if (materialData._material == material)
					{
						return true;
					}
				}
			}
			return false;
		}

		/// <summary>
		/// Adds gameobjects that should be cloned when cloning the whole asset.
		/// </summary>
		/// <param name="clonableObjects">List of game objects to add to</param>
		public void GetClonableObjects(List<GameObject> clonableObjects)
		{
			// TODO: check if geotype not HAPI_GeoType.HAPI_GEOTYPE_INTERMEDIATE

			if (!IsPartInstanced() && _gameObject != null)
			{
				clonableObjects.Add(_gameObject);
			}
		}

		public void GetClonableParts(List<HEU_PartData> clonableParts)
		{
			if (!IsPartInstanced() && _gameObject != null)
			{
				clonableParts.Add(this);
			}
		}

		/// <summary>
		/// Adds gameobjects that were output from this part.
		/// </summary>
		/// <param name="outputObjects">List to add to</param>
		public void GetOutputGameObjects(List<GameObject> outputObjects)
		{
			// TODO: check if geotype not HAPI_GeoType.HAPI_GEOTYPE_INTERMEDIATE

			if (!IsPartInstanced() && _gameObject != null)
			{
				outputObjects.Add(_gameObject);
			}
		}

		/// <summary>
		/// Returns self if it has the given output gameobject.
		/// </summary>
		/// <param name="outputGameObject">The output gameobject to check</param>
		/// <returns>Valid HEU_PartData or null if no match</returns>
		public HEU_PartData GetHDAPartWithGameObject(GameObject outputGameObject)
		{
			return (outputGameObject == _gameObject) ? this : null;
		}

		private void SetObjectInstancer(bool bObjectInstancer)
		{
			_isObjectInstancer = bObjectInstancer;
		}

        /// <summary>
        /// Clear out existing instances for this part.
        /// </summary>
        public void ClearInstances()
        {
            List<GameObject> instances = HEU_GeneralUtility.GetInstanceChildObjects(this._gameObject);
            for (int i = 0; i < instances.Count; ++i)
            {
				HEU_GeneralUtility.DestroyGeneratedComponents(instances[i]);
                HEU_GeneralUtility.DestroyImmediate(instances[i]);
            }

			_haveInstancesBeenGenerated = false;
		}

		/// <summary>
		/// Clear out object instance infos for this part.
		/// </summary>
		private void ClearObjectInstanceInfos()
		{
			if (_objectInstanceInfos != null)
			{
				int numObjInstances = _objectInstanceInfos.Count;
				for (int i = 0; i < numObjInstances; ++i)
				{
					HEU_GeneralUtility.DestroyImmediate(_objectInstanceInfos[i]);
				}
				_objectInstanceInfos.Clear();

				ObjectInstancesBeenGenerated = false;
			}
		}

		/// <summary>
		/// Clean up and remove any HEU_ObjectInstanceInfos that don't have 
		/// valid parts. This can happen if the object node being instanced
		/// has changed (no parts). The instancer should then clear out 
		/// any created HEU_ObjectInstanceInfos for that object node as otherwise
		/// it leaves a dangling instance input for the user.
		/// </summary>
		public void ClearInvalidObjectInstanceInfos()
		{
			if (_objectInstanceInfos != null)
			{
				int numObjInstances = _objectInstanceInfos.Count;
				for (int i = 0; i < numObjInstances; ++i)
				{
					// Presume that if invalid ID then this is using Unity object instead of Houdini generated object
					if(_objectInstanceInfos[i]._instancedObjectNodeID == HEU_Defines.HEU_INVALID_NODE_ID)
					{
						continue;
					}

					bool bDestroyIt = true;
					HEU_ObjectNode instancedObjNode = ParentAsset.GetObjectWithID(_objectInstanceInfos[i]._instancedObjectNodeID);
					if (instancedObjNode != null)
					{
						List<HEU_PartData> cloneParts = new List<HEU_PartData>();
						instancedObjNode.GetClonableParts(cloneParts);
						bDestroyIt = cloneParts.Count == 0;
					}

					if(bDestroyIt)
					{
						HEU_ObjectInstanceInfo objInstanceInfo = _objectInstanceInfos[i];
						_objectInstanceInfos.RemoveAt(i);
						i--;
						numObjInstances = _objectInstanceInfos.Count;

						HEU_GeneralUtility.DestroyImmediate(objInstanceInfo);
					}
				}
			}
		}

		/// <summary>
		/// Clear generated data for this part.
		/// </summary>
		public void ClearGeneratedData()
		{
			ClearInstances();
			HEU_GeneralUtility.DestroyGeneratedComponents(_gameObject);

			ObjectInstancesBeenGenerated = false;
		}

		/// <summary>
		/// Generate part instances (packed primvites).
		/// </summary>
		public void GeneratePartInstances(HEU_SessionBase session)
		{
			if(HaveInstancesBeenGenerated())
			{
				Debug.LogWarningFormat("Part {0} has already had its instances generated!", name);
				return;
			}

			HAPI_PartInfo partInfo = new HAPI_PartInfo();
			if (!session.GetPartInfo(_geoID, _partID, ref partInfo))
			{
				return;
			}

			//Debug.LogFormat("Instancer: name={0}, instanced={1}, instance count={2}, instance part count={3}",
			//	HEU_SessionManager.GetString(partInfo.nameSH, session), partInfo.isInstanced, partInfo.instanceCount, partInfo.instancedPartCount);

			if (!IsPartInstancer())
			{
				Debug.LogErrorFormat("Generate Part Instances called on a non-instancer part {0} for asset {1}!", PartName, ParentAsset.AssetName);
				return;
			}

			if (partInfo.instancedPartCount <= 0)
			{
				Debug.LogErrorFormat("Invalid instanced part count: {0} for part {1} of asset {2}", partInfo.instancedPartCount, PartName, ParentAsset.AssetName);
				return;
			}

			// Get the instance node IDs to get the geometry to be instanced.
			// Get the instanced count to all the instances. These will end up being mesh references to the mesh from instance node IDs.

			Transform partTransform = this._gameObject.transform;

			// Get each instance's transform
			HAPI_Transform[] instanceTransforms = new HAPI_Transform[partInfo.instanceCount];
			if (!session.GetInstancerPartTransforms(_geoID, PartID, HAPI_RSTOrder.HAPI_SRT, instanceTransforms, 0, partInfo.instanceCount))
			{
				return;
			}

			// Get part IDs for the parts being instanced
			HAPI_NodeId[] instanceNodeIDs = new HAPI_NodeId[partInfo.instancedPartCount];
			if (!session.GetInstancedPartIds(_geoID, PartID, instanceNodeIDs, 0, partInfo.instancedPartCount))
			{
				return;
			}

			int numInstances = instanceNodeIDs.Length;
			for (int i = 0; i < numInstances; ++i)
			{
				HEU_PartData partData = _geoNode.GetPartFromPartID(instanceNodeIDs[i]);
				if (partData == null)
				{
					Debug.LogErrorFormat("Part with id {0} is missing. Unable to setup instancer!", instanceNodeIDs[i]);
					return;
				}

				// If the part we're instancing is itself an instancer, make sure it has generated its instances
				if(partData.IsPartInstancer() && !partData.HaveInstancesBeenGenerated())
				{
					partData.GeneratePartInstances(session);
				}

				Debug.Assert(partData._gameObject != null, "Instancer's reference (part) is missing gameobject!");

				HAPI_PartInfo instancePartInfo = new HAPI_PartInfo();
				session.GetPartInfo(_geoID, instanceNodeIDs[i], ref instancePartInfo);

				int numTransforms = instanceTransforms.Length;
				for (int j = 0; j < numTransforms; ++j)
				{
					GameObject newInstanceGO = HEU_EditorUtility.InstantiateGameObject(partData._gameObject, partTransform, false, false);
					newInstanceGO.name = PartName + HEU_Defines.HEU_INSTANCE_NAME + (j + 1);

					newInstanceGO.isStatic = this._gameObject.isStatic;

					HEU_HAPIUtility.ApplyLocalTransfromFromHoudiniToUnity(ref instanceTransforms[j], newInstanceGO.transform);

					// When cloning, the instanced part might have been made invisible, so re-enable renderer to have the cloned instance display it.
					HEU_GeneralUtility.SetGameObjectRenderVisiblity(newInstanceGO, true);
				}
			}

			_haveInstancesBeenGenerated = true;
		}

		/// <summary>
		/// Generate instances from given Houdini Engine object node ID
		/// </summary>
		/// <param name="session">Active session to use</param>
		/// <param name="objectNodeID">The source object node ID to create instances from</param>
		public void GenerateInstancesFromObjectID(HEU_SessionBase session, HAPI_NodeId objectNodeID)
		{
			int numInstances = GetPartPointCount();
			if (numInstances <= 0)
			{
				return;
			}

			HEU_ObjectInstanceInfo instanceInfo = GetObjectInstanceInfoWithObjectID(objectNodeID);
			if (instanceInfo != null && (instanceInfo._instancedInputs.Count > 0))
			{
				List<HEU_InstancedInput> validInstancedGameObjects = instanceInfo._instancedInputs;
				int instancedObjCount = validInstancedGameObjects.Count;

				SetObjectInstancer(true);
				ObjectInstancesBeenGenerated = true;

				Transform partTransform = this._gameObject.transform;

				HAPI_Transform[] instanceTransforms = new HAPI_Transform[numInstances];
				if (session.GetInstanceTransforms(_geoID, HAPI_RSTOrder.HAPI_SRT, instanceTransforms, 0, numInstances))
				{
					int numTransforms = instanceTransforms.Length;
					for (int j = 0; j < numTransforms; ++j)
					{
						int randomIndex = UnityEngine.Random.Range(0, instancedObjCount);
						CreateNewInstanceFromObject(validInstancedGameObjects[randomIndex]._instancedGameObject, (j + 1), partTransform, 
							ref instanceTransforms[j], objectNodeID, null, 
							validInstancedGameObjects[randomIndex]._rotationOffset, validInstancedGameObjects[randomIndex]._scaleOffset);
					}
				}
			}
			else
			{
				HEU_ObjectNode instancedObjNode = ParentAsset.GetObjectWithID(objectNodeID);
				if (instancedObjNode != null)
				{
					GenerateInstancesFromObject(session, instancedObjNode);
				}
				else
				{
					Debug.LogWarningFormat("Instanced object with ID {0} not found. Unable to generate instances!", objectNodeID);
				}
			}
		}

		/// <summary>
		/// Generate instances from another object node (sourceObject).
		/// </summary>
		/// <param name="session"></param>
		/// <param name="sourceObject">The object node to create instances from.</param>
		public void GenerateInstancesFromObject(HEU_SessionBase session, HEU_ObjectNode sourceObject)
		{
			// Create instance of this object for all points

			List<HEU_PartData> clonableParts = new List<HEU_PartData>();
			sourceObject.GetClonableParts(clonableParts);

			int numInstances = GetPartPointCount();
			if (numInstances <= 0)
			{
				return;
			}

			SetObjectInstancer(true);
			ObjectInstancesBeenGenerated = true;

			Transform partTransform = this._gameObject.transform;

			HAPI_Transform[] instanceTransforms = new HAPI_Transform[numInstances];
			if (session.GetInstanceTransforms(_geoID, HAPI_RSTOrder.HAPI_SRT, instanceTransforms, 0, numInstances))
			{
				int numInstancesCreated = 0;
				int numTransforms = instanceTransforms.Length;
				for (int j = 0; j < numTransforms; ++j)
				{
					int numClones = clonableParts.Count;
					for (int c = 0; c < numClones; ++c)
					{
						CreateNewInstanceFromObject(clonableParts[c]._gameObject, (numInstancesCreated + 1), partTransform, ref instanceTransforms[j], 
							sourceObject.ObjectID, null, Vector3.zero, Vector3.one);
						numInstancesCreated++;
					}
				}
			}
		}

		/// <summary>
		/// Generate instances from object IDs found in the asset.
		/// </summary>
		/// <param name="session"></param>
		public void GenerateInstancesFromObjectIds(HEU_SessionBase session)
		{
			int numInstances = GetPartPointCount();
			if (numInstances <= 0)
			{
				return;
			}

			HAPI_NodeId[] instancedNodeIds = new HAPI_NodeId[numInstances];
			if (!session.GetInstancedObjectIds(_geoID, instancedNodeIds, 0, numInstances))
			{
				return;
			}

			HAPI_Transform[] instanceTransforms = new HAPI_Transform[numInstances];
			if (!session.GetInstanceTransforms(_geoID, HAPI_RSTOrder.HAPI_SRT, instanceTransforms, 0, numInstances))
			{
				return;
			}

			SetObjectInstancer(true);
			ObjectInstancesBeenGenerated = true;

			Transform partTransform = this._gameObject.transform;

			int numInstancesCreated = 0;
			for (int i = 0; i < numInstances; ++i)
			{
				if (instancedNodeIds[i] == HEU_Defines.HEU_INVALID_NODE_ID)
				{
					// Skipping points without valid instanced IDs
					continue;
				}

				HEU_ObjectInstanceInfo instanceInfo = GetObjectInstanceInfoWithObjectID(instancedNodeIds[i]);
				if(instanceInfo != null && (instanceInfo._instancedInputs.Count > 0))
				{
					List<HEU_InstancedInput> validInstancedGameObjects = instanceInfo._instancedInputs;
					int randomIndex = UnityEngine.Random.Range(0, validInstancedGameObjects.Count);

					CreateNewInstanceFromObject(validInstancedGameObjects[randomIndex]._instancedGameObject, (numInstancesCreated + 1), partTransform, ref instanceTransforms[i],
						instanceInfo._instancedObjectNodeID, null, 
						validInstancedGameObjects[randomIndex]._rotationOffset, validInstancedGameObjects[randomIndex]._scaleOffset);
					numInstancesCreated++;
				}
				else
				{
					HEU_ObjectNode instancedObjNode = ParentAsset.GetObjectWithID(instancedNodeIds[i]);
					if (instancedObjNode == null)
					{
						Debug.LogErrorFormat("Object with ID {0} not found for instancing!", instancedNodeIds[i]);
						continue;
					}

					List<HEU_PartData> cloneParts = new List<HEU_PartData>();
					instancedObjNode.GetClonableParts(cloneParts);

					int numClones = cloneParts.Count;
					for (int c = 0; c < numClones; ++c)
					{
						CreateNewInstanceFromObject(cloneParts[c]._gameObject, (numInstancesCreated + 1), partTransform, ref instanceTransforms[i],
							instancedObjNode.ObjectID, null, Vector3.zero, Vector3.one);
						numInstancesCreated++;
					}
				}
			}
		}

		/// <summary>
		/// Generate instances from Unity objects specified via attributes. 
		/// </summary>
		/// <param name="session"></param>
		/// <param name="unityInstanceAttr">Name of the attribute to get the Unity path from.</param>
		public void GenerateInstancesFromUnityAssetPathAttribute(HEU_SessionBase session, string unityInstanceAttr)
		{
			int numInstances = GetPartPointCount();
			if (numInstances <= 0)
			{
				return;
			}

			HAPI_Transform[] instanceTransforms = new HAPI_Transform[numInstances];
			if (!session.GetInstanceTransforms(_geoID, HAPI_RSTOrder.HAPI_SRT, instanceTransforms, 0, numInstances))
			{
				return;
			}

			HAPI_AttributeInfo instanceAttrInfo = new HAPI_AttributeInfo();
			int[] instanceAttrID = new int[0];
			HEU_GeneralUtility.GetAttribute(session, _geoID, _partID, unityInstanceAttr, ref instanceAttrInfo, ref instanceAttrID, session.GetAttributeStringData);

			string[] instancePathAttrValues = HEU_SessionManager.GetStringValuesFromStringIndices(instanceAttrID);

			Debug.AssertFormat(instanceAttrInfo.owner == HAPI_AttributeOwner.HAPI_ATTROWNER_POINT, "Expected to parse {0} owner attribute but got {1} instead!", HAPI_AttributeOwner.HAPI_ATTROWNER_POINT, instanceAttrInfo.owner);
			Debug.AssertFormat(instancePathAttrValues.Length == numInstances, "Number of instances {0} does not match point attribute count {1} for part {2} of asset {3}",
				numInstances, instancePathAttrValues.Length, PartName, ParentAsset.AssetName);

			SetObjectInstancer(true);
			ObjectInstancesBeenGenerated = true;

			Transform partTransform = this._gameObject.transform;

			// Keep track of loaded objects so we only need to load once for each object
			Dictionary<string, GameObject> loadedUnityObjectMap = new Dictionary<string, GameObject>();

			// Temporary empty gameobject in case where specified Unity object is not found
			GameObject tempGO = null;

			Dictionary<HEU_ObjectInstanceInfo, List<HEU_InstancedInput>> validInstancedGameObjectsMap = new Dictionary<HEU_ObjectInstanceInfo, List<HEU_InstancedInput>>();

			int numInstancesCreated = 0;
			for (int i = 0; i < numInstances; ++i)
			{
				GameObject unitySrcGO = null;

				Vector3 rotationOffset = Vector3.zero;
				Vector3 scaleOffset = Vector3.one;

				HEU_ObjectInstanceInfo instanceInfo = GetObjectInstanceInfoWithObjectPath(instancePathAttrValues[i]);
				if (instanceInfo != null && (instanceInfo._instancedInputs.Count > 0))
				{
					List<HEU_InstancedInput> validInstancedGameObjects = instanceInfo._instancedInputs;
					int randomIndex = UnityEngine.Random.Range(0, validInstancedGameObjects.Count);

					unitySrcGO = validInstancedGameObjects[randomIndex]._instancedGameObject;
					rotationOffset = validInstancedGameObjects[randomIndex]._rotationOffset;
					scaleOffset = validInstancedGameObjects[randomIndex]._scaleOffset;
				}

				if (unitySrcGO == null)
				{
					if (string.IsNullOrEmpty(instancePathAttrValues[i]))
					{
						continue;
					}

					if (!loadedUnityObjectMap.TryGetValue(instancePathAttrValues[i], out unitySrcGO))
					{
						// Try loading it
						HEU_AssetDatabase.ImportAsset(instancePathAttrValues[i], HEU_AssetDatabase.HEU_ImportAssetOptions.Default);
						unitySrcGO = HEU_AssetDatabase.LoadAssetAtPath(instancePathAttrValues[i], typeof(GameObject)) as GameObject;

						if (unitySrcGO == null)
						{
							Debug.LogErrorFormat("Unable to load asset at {0} for instancing!", instancePathAttrValues[i]);

							// Even though the source Unity object is not found, we should create an object instance info so
							// that it will be exposed in UI and user can override
							if (tempGO == null)
							{
								tempGO = new GameObject();
							}
							unitySrcGO = tempGO;
						}

						// Adding to map even if not found so we don't flood the log with the same error message
						loadedUnityObjectMap.Add(instancePathAttrValues[i], unitySrcGO);
					}
				}

				CreateNewInstanceFromObject(unitySrcGO, (numInstancesCreated + 1), partTransform, ref instanceTransforms[i], 
					HEU_Defines.HEU_INVALID_NODE_ID, instancePathAttrValues[i], rotationOffset, scaleOffset);
				numInstancesCreated++;
			}

			if (tempGO != null)
			{
				HEU_GeneralUtility.DestroyImmediate(tempGO, bRegisterUndo:false);
			}
		}

		/// <summary>
		/// Generate instances from a single existing Unity asset.
		/// </summary>
		/// <param name="session"></param>
		/// <param name="assetPath"></param>
		public void GenerateInstancesFromUnityAssetPath(HEU_SessionBase session, string assetPath)
		{
			int numInstances = GetPartPointCount();
			if (numInstances <= 0)
			{
				return;
			}

			HAPI_Transform[] instanceTransforms = new HAPI_Transform[numInstances];
			if (!session.GetInstanceTransforms(_geoID, HAPI_RSTOrder.HAPI_SRT, instanceTransforms, 0, numInstances))
			{
				return;
			}

			SetObjectInstancer(true);
			ObjectInstancesBeenGenerated = true;

			GameObject instancedAssetGameObject = null;

			HEU_ObjectInstanceInfo instanceInfo = GetObjectInstanceInfoWithObjectPath(assetPath);

			List<HEU_InstancedInput> validInstancedGameObjects = null;
			int instancedObjCount = 0;

			Vector3 rotationOffset = Vector3.zero;
			Vector3 scaleOffset = Vector3.one;

			if (instanceInfo != null && (instanceInfo._instancedInputs.Count > 0))
			{
				validInstancedGameObjects = instanceInfo._instancedInputs;
				instancedObjCount = validInstancedGameObjects.Count;
			}
			
			if(instancedObjCount == 0)
			{
				HEU_AssetDatabase.ImportAsset(assetPath, HEU_AssetDatabase.HEU_ImportAssetOptions.Default);
				instancedAssetGameObject = HEU_AssetDatabase.LoadAssetAtPath(assetPath, typeof(GameObject)) as GameObject;
			}

			if (instancedAssetGameObject != null)
			{
				int numInstancesCreated = 0;
				for (int i = 0; i < numInstances; ++i)
				{
					GameObject instancedGameObject;
					if(instancedAssetGameObject == null)
					{
						// Get random override
						int randomIndex = UnityEngine.Random.Range(0, instancedObjCount);
						instancedGameObject = validInstancedGameObjects[randomIndex]._instancedGameObject;
						rotationOffset = validInstancedGameObjects[randomIndex]._rotationOffset;
						scaleOffset = validInstancedGameObjects[randomIndex]._scaleOffset;
					}
					else
					{
						instancedGameObject = instancedAssetGameObject;
					}

					CreateNewInstanceFromObject(instancedGameObject, (numInstancesCreated + 1), this._gameObject.transform, ref instanceTransforms[i], 
						HEU_Defines.HEU_INVALID_NODE_ID, assetPath, rotationOffset, scaleOffset);
					numInstancesCreated++;
				}
			}
			else
			{
				Debug.LogErrorFormat("Unable to load asset at {0} for instancing!", assetPath);
			}
		}

		/// <summary>
		/// Create a new instance of the sourceObject.
		/// </summary>
		/// <param name="sourceObject">GameObject to instance.</param>
		/// <param name="instanceIndex">Index of the instance within the part.</param>
		/// <param name="parentTransform">Parent of the new instance.</param>
		/// <param name="hapiTransform">HAPI transform to apply to the new instance.</param>
		private void CreateNewInstanceFromObject(GameObject sourceObject, int instanceIndex, Transform parentTransform, ref HAPI_Transform hapiTransform, 
			HAPI_NodeId instancedObjectNodeID, string instancedObjectPath, Vector3 rotationOffset, Vector3 scaleOffset)
		{
			GameObject newInstanceGO = null;

			if (HEU_EditorUtility.IsPrefabOriginal(sourceObject))
			{
				newInstanceGO = HEU_EditorUtility.InstantiatePrefab(sourceObject) as GameObject;
				newInstanceGO.transform.parent = parentTransform;
			}
			else
			{
				newInstanceGO = HEU_EditorUtility.InstantiateGameObject(sourceObject, parentTransform, false, false);	
			}
			
			newInstanceGO.name = PartName + HEU_Defines.HEU_INSTANCE_NAME + instanceIndex;

			newInstanceGO.isStatic = this._gameObject.isStatic;

			Transform instanceTransform = newInstanceGO.transform;
			HEU_HAPIUtility.ApplyLocalTransfromFromHoudiniToUnity(ref hapiTransform, instanceTransform);

			// Apply offsets
			Vector3 rotation = instanceTransform.localRotation.eulerAngles;
			instanceTransform.localRotation = Quaternion.Euler(rotation + rotationOffset);
			instanceTransform.localScale = Vector3.Scale(instanceTransform.localScale, scaleOffset);

			// When cloning, the instanced part might have been made invisible, so re-enable renderer to have the cloned instance display it.
			HEU_GeneralUtility.SetGameObjectRenderVisiblity(newInstanceGO, true);

			// Add to object instance info map. Find existing object instance info, or create it.
			HEU_ObjectInstanceInfo instanceInfo = null;
			if(instancedObjectNodeID != HEU_Defines.HEU_INVALID_NODE_ID)
			{
				instanceInfo = GetObjectInstanceInfoWithObjectID(instancedObjectNodeID);
			}
			else if(!string.IsNullOrEmpty(instancedObjectPath))
			{
				instanceInfo = GetObjectInstanceInfoWithObjectPath(instancedObjectPath);
			}

			if (instanceInfo == null)
			{
				instanceInfo = CreateObjectInstanceInfo(sourceObject, instancedObjectNodeID, instancedObjectPath);
			}

			instanceInfo._instances.Add(newInstanceGO);
		}

		public HEU_Curve GetCurve(bool bEditableOnly)
		{
			if(_curve != null && (!bEditableOnly || _curve.IsEditable()))
			{
				return _curve;
			}
			return null;
		}

		/// <summary>
		/// Set visibility on this part's gameobject.
		/// </summary>
		/// <param name="bVisibility">True if visible.</param>
		public void SetVisiblity(bool bVisibility)
		{
			if(_curve != null)
			{
				bVisibility &= HEU_PluginSettings.Curves_ShowInSceneView;
			}
			HEU_GeneralUtility.SetGameObjectRenderVisiblity(_gameObject, bVisibility);
		}

		/// <summary>
		/// Calculate the visiblity of this parent, based on parent's state and part properties.
		/// </summary>
		/// <param name="bParentVisibility">True if parent is visible</param>
		public void CalculateVisibility(bool bParentVisibility)
		{
			bool bIsVisible = !IsPartInstanced() && bParentVisibility && !_isPartEditable;
			SetVisiblity(bIsVisible);
		}

		/// <summary>
		/// Copy relevant components from sourceGO to targetGO.
		/// </summary>
		/// <param name="sourceGO">Source gameobject to copy from.</param>
		/// <param name="targetGO">Target gameobject to copy to.</param>
		/// <param name="assetName">Name of the asset.</param>
		/// <param name="sourceToTargetMeshMap">Map of existing meshes to newly created meshes. This helps keep track of shared meshes that should be copied but still shared in new asset.</param>
		/// <param name="bWriteMeshesToAssetDatabase">Whether to store meshes to database. Required for prefabs.</param>
		/// <param name="bakedAssetPath">Path to asset's database cache. Could be null in which case it will be filled.</param>
		/// <param name="assetDBObject">The asset database object to write out the persistent mesh data to. Could be null, in which case it might be created.</param>
		/// <param name="assetObjectFileName">File name of the asset database object. This will be used to create new assetDBObject.</param>
		private void CopyGameObjectComponents(GameObject sourceGO, GameObject targetGO, string assetName, Dictionary<Mesh, Mesh> sourceToTargetMeshMap, bool bWriteMeshesToAssetDatabase, ref string bakedAssetPath, ref UnityEngine.Object assetDBObject, string assetObjectFileName)
		{
			// Copy mesh, collider, material, and textures into its own directory in the Assets folder

			// Mesh for render
			MeshFilter targetMeshFilter = targetGO.GetComponent<MeshFilter>();
			MeshFilter sourceMeshFilter = sourceGO.GetComponent<MeshFilter>();
			if (sourceMeshFilter != null)
			{
				if (targetMeshFilter == null)
				{
					targetMeshFilter = HEU_EditorUtility.AddComponent<MeshFilter>(targetGO, true) as MeshFilter;
				}

				Mesh originalMesh = sourceMeshFilter.sharedMesh;
				if (originalMesh != null)
				{
					Mesh targetMesh = null;
					if (!sourceToTargetMeshMap.TryGetValue(originalMesh, out targetMesh))
					{
						// Create this mesh
						targetMesh = Mesh.Instantiate(originalMesh) as Mesh;
						sourceToTargetMeshMap[originalMesh] = targetMesh;

						if (bWriteMeshesToAssetDatabase)
						{
							HEU_AssetDatabase.CreateAddObjectInAssetCacheFolder(assetName, assetObjectFileName, targetMesh, ref bakedAssetPath, ref assetDBObject);
						}
					}

					targetMeshFilter.sharedMesh = targetMesh;
				}
			}
			else if (targetMeshFilter != null)
			{
				HEU_GeneralUtility.DestroyImmediate(targetMeshFilter);
			}

			// Mesh for collider
			MeshCollider targetMeshCollider = targetGO.GetComponent<MeshCollider>();
			MeshCollider sourceMeshCollider = sourceGO.GetComponent<MeshCollider>();
			if (sourceMeshCollider != null)
			{
				if (targetMeshCollider == null)
				{
					targetMeshCollider = HEU_EditorUtility.AddComponent<MeshCollider>(targetGO, true) as MeshCollider;
				}

				Mesh originalColliderMesh = sourceMeshCollider.sharedMesh;
				if (originalColliderMesh != null)
				{
					Mesh targetColliderMesh = null;
					if (!sourceToTargetMeshMap.TryGetValue(originalColliderMesh, out targetColliderMesh))
					{
						// Create this mesh
						targetColliderMesh = Mesh.Instantiate(originalColliderMesh) as Mesh;
						sourceToTargetMeshMap[originalColliderMesh] = targetColliderMesh;

						if (bWriteMeshesToAssetDatabase)
						{
							HEU_AssetDatabase.CreateAddObjectInAssetCacheFolder(assetName, assetObjectFileName, targetColliderMesh, ref bakedAssetPath, ref assetDBObject);
						}
					}

					targetMeshCollider.sharedMesh = targetColliderMesh;
				}
			}
			else if (targetMeshCollider != null)
			{
				HEU_GeneralUtility.DestroyImmediate(targetMeshFilter);
			}

			// Materials and textures
			MeshRenderer targetMeshRenderer = targetGO.GetComponent<MeshRenderer>();
			MeshRenderer sourceMeshRenderer = sourceGO.GetComponent<MeshRenderer>();
			if (sourceMeshRenderer != null)
			{
				if (targetMeshRenderer == null)
				{
					targetMeshRenderer = HEU_EditorUtility.AddComponent<MeshRenderer>(targetGO, true) as MeshRenderer;
				}

				Material[] materials = sourceMeshRenderer.sharedMaterials;
				if (materials != null && materials.Length > 0)
				{
					if (string.IsNullOrEmpty(bakedAssetPath))
					{
						// Need to create the baked folder in order to store materials and textures
						bakedAssetPath = HEU_AssetDatabase.CreateUniqueBakePath(assetName);
					}

					int numMaterials = materials.Length;
					for (int m = 0; m < numMaterials; ++m)
					{
						Material srcMaterial = materials[m];

						string materialPath = HEU_AssetDatabase.GetAssetPath(srcMaterial);
						if (!string.IsNullOrEmpty(materialPath) && HEU_AssetDatabase.IsPathInAssetCache(materialPath))
						{
							Material newMaterial = HEU_AssetDatabase.LoadAssetCopy(srcMaterial, bakedAssetPath, typeof(Material)) as Material;
							if (newMaterial == null)
							{
								throw new HEU_HoudiniEngineError(string.Format("Unable to copy material. Stopping bake!"));
							}

							// Diffuse texture
							if (newMaterial.HasProperty("_MainTex"))
							{
								Texture srcDiffuseTexture = newMaterial.mainTexture;
								if (srcDiffuseTexture != null)
								{
									Texture newDiffuseTexture = HEU_AssetDatabase.LoadAssetCopy(srcDiffuseTexture, bakedAssetPath, typeof(Texture)) as Texture;
									if (newDiffuseTexture == null)
									{
										throw new HEU_HoudiniEngineError(string.Format("Unable to copy texture. Stopping bake!"));
									}
									newMaterial.mainTexture = newDiffuseTexture;
								}
							}

							// Normal map
							Texture srcNormalMap = materials[m].GetTexture(HEU_Defines.UNITY_SHADER_BUMP_MAP);
							if (srcNormalMap != null)
							{
								Texture newNormalMap = HEU_AssetDatabase.LoadAssetCopy(srcNormalMap, bakedAssetPath, typeof(Texture)) as Texture;
								if (newNormalMap == null)
								{
									throw new HEU_HoudiniEngineError(string.Format("Unable to copy texture. Stopping bake!"));
								}
								newMaterial.SetTexture(HEU_Defines.UNITY_SHADER_BUMP_MAP, newNormalMap);
							}

							materials[m] = newMaterial;
						}
					}

					targetMeshRenderer.sharedMaterials = materials;
				}
			}
			else if (targetMeshRenderer != null)
			{
				HEU_GeneralUtility.DestroyImmediate(targetMeshRenderer);
			}
		}

		/// <summary>
		/// Bake this part out to a new gameobject, and returns it. 
		/// Copies all relevant components.
		/// Supports baking of part and object instances.
		/// </summary>
		/// <param name="parentTransform">The parent for the new object. Can be null.</param>
		/// <param name="bWriteMeshesToAssetDatabase">Whether to store meshes to database. Required for prefabs.</param>
		/// <param name="bakedAssetPath">Path to asset's database cache. Could be null in which case it will be filled.</param>
		/// <param name="sourceToTargetMeshMap">Map of existing meshes to newly created meshes. This helps keep track of shared meshes that should be copied but still shared in new asset.</param>
		/// <param name="assetDBObject">The asset database object to write out the persistent mesh data to. Could be null, in which case it might be created.</param>
		/// <param name="assetObjectFileName">File name of the asset database object. This will be used to create new assetDBObject.</param>
		/// <param name="bReconnectPrefabInstances">Reconnect prefab instances to its prefab parent.</param>
		/// <returns>The newly created gameobject.</returns>
		public GameObject BakePartToNewGameObject(Transform parentTransform, bool bWriteMeshesToAssetDatabase, ref string bakedAssetPath, Dictionary<Mesh, Mesh> sourceToTargetMeshMap, ref UnityEngine.Object assetDBObject, string assetObjectFileName, bool bReconnectPrefabInstances)
		{
			if (_gameObject == null)
			{
				return null;
			}
			// This creates a copy of the part's gameobject, along with instances if it has them.
			// If the instances are prefab instances, then this disconnects the connection. We re-connect them back in the call below.
			GameObject targetGO = HEU_EditorUtility.InstantiateGameObject(_gameObject, parentTransform, true, true);
			targetGO.name = HEU_PartData.AppendBakedCloneName(_gameObject.name);

			BakePartToGameObject(targetGO, false, false, bWriteMeshesToAssetDatabase, ref bakedAssetPath, sourceToTargetMeshMap, ref assetDBObject, assetObjectFileName, bReconnectPrefabInstances);

			return targetGO;
		}

		/// <summary>
		/// Bake this part out to the given targetGO. Existing components might be destroyed.
		/// Supports baking of part and object instances.
		/// </summary>
		/// <param name="targetGO">Target gameobject to bake out to.</param>
		/// <param name="bDeleteExistingComponents">Whether to destroy existing components on the targetGO.</param>
		/// <param name="bDontDeletePersistantResources">Whether to delete persistant resources stored in the project.</param>
		/// <param name="bWriteMeshesToAssetDatabase">Whether to store meshes to database. Required for prefabs.</param>
		/// <param name="bakedAssetPath">Path to asset's database cache. Could be null in which case it will be filled.</param>
		/// <param name="sourceToTargetMeshMap">Map of existing meshes to newly created meshes. This helps keep track of shared meshes that should be copied but still shared in new asset.</param>
		/// <param name="assetDBObject">The asset database object to write out the persistent mesh data to. Could be null, in which case it might be created.</param>
		/// <param name="assetObjectFileName">File name of the asset database object. This will be used to create new assetDBObject.</param>
		/// <param name="bReconnectPrefabInstances">Reconnect prefab instances to its prefab parent.</param>
		public void BakePartToGameObject(GameObject targetGO, bool bDeleteExistingComponents, bool bDontDeletePersistantResources, bool bWriteMeshesToAssetDatabase, ref string bakedAssetPath, Dictionary<Mesh, Mesh> sourceToTargetMeshMap, ref UnityEngine.Object assetDBObject, string assetObjectFileName, bool bReconnectPrefabInstances)
		{
			if (_gameObject == null)
			{
				return;
			}
			else if (_gameObject == targetGO)
			{
				Debug.LogError("Copy and target objects cannot be the same!");
				return;
			}

			string assetName = ParentAsset.AssetName;

			Transform targetTransform = targetGO.transform;

			// Keeps track of unprocessed children. Any leftover will be destroyed.
			List<GameObject> unprocessedTargetChildren = HEU_GeneralUtility.GetChildGameObjects(targetGO);

			if (IsPartInstancer() || IsObjectInstancer())
			{
				// Instancer

				// Instancer has a gameobject with children. The parent is an empty transform, while the
				// the children have all the data. The children could have an assortment of meshes.

				List<GameObject> srcChildGameObjects = HEU_GeneralUtility.GetChildGameObjects(_gameObject);
				int numChildren = srcChildGameObjects.Count;
				for (int i = 0; i < numChildren; ++i)
				{
					GameObject srcChildGO = srcChildGameObjects[i];

					GameObject targetChildGO = HEU_GeneralUtility.GetGameObjectByName(unprocessedTargetChildren, srcChildGO.name);
					if (targetChildGO == null)
					{
						targetChildGO = new GameObject(srcChildGO.name);
						targetChildGO.transform.parent = targetTransform;
					}
					else
					{
						if (bDeleteExistingComponents)
						{
							HEU_PartData.DestroyExistingGeneratedComponentsMeshData(targetChildGO, bDontDeletePersistantResources);
						}

						unprocessedTargetChildren.Remove(targetChildGO);

						// Update transform of each existing instance
						HEU_GeneralUtility.CopyLocalTransformValues(srcChildGO.transform, targetChildGO.transform);

						if (bReconnectPrefabInstances && HEU_EditorUtility.IsPrefabInstance(srcChildGO))
						{
							// Reconnect back to the prefab if the source was a prefab instance
							GameObject prefabSource = HEU_EditorUtility.GetPrefabParent(srcChildGO) as GameObject;
							if (prefabSource != null)
							{
								targetChildGO = HEU_EditorUtility.ConnectGameObjectToPrefab(targetChildGO, prefabSource);

								// Update transform of each existing instance again since prefab connect above resets it
								HEU_GeneralUtility.CopyLocalTransformValues(srcChildGO.transform, targetChildGO.transform);

								continue;
							}
						}
					}

					// Copy component data
					CopyGameObjectComponents(srcChildGO, targetChildGO, assetName, sourceToTargetMeshMap, bWriteMeshesToAssetDatabase, ref bakedAssetPath, ref assetDBObject, assetObjectFileName);
				}
			}
			else
			{
				// Not an instancer, regular object (could also be instanced)
				// TODO: For instanced object, should we not instantiate if it is not visible?

				if (bDeleteExistingComponents)
				{
					HEU_PartData.DestroyExistingGeneratedComponentsMeshData(targetGO, bDontDeletePersistantResources);
				}

				// Copy component data
				CopyGameObjectComponents(_gameObject, targetGO, assetName, sourceToTargetMeshMap, bWriteMeshesToAssetDatabase, ref bakedAssetPath, ref assetDBObject, assetObjectFileName);
			}

			if (unprocessedTargetChildren.Count > 0)
			{
				// Clean up any children that we haven't updated as they don't exist in the source
				HEU_GeneralUtility.DestroyBakedGameObjects(unprocessedTargetChildren);
			}
		}

		/// <summary>
		/// Destroy existing components on targetGO which were generated by our bake process.
		/// Persistent resources like meshes, materials, textures will be deleted if bDontDeletePersistentResources is true.
		/// Fills in targetAssetPath with targetGO's asset cache path.
		/// </summary>
		/// <param name="targetGO">The gameobject to destroy components of</param>
		/// <param name="bDontDeletePersistantResources">Whether to delete persistant data</param>
		/// <param name="targetAssetPath">targetGO's asset cache path, if used</param>
		public static void DestroyExistingGeneratedComponentsMeshData(GameObject targetGO, bool bDontDeletePersistantResources)
		{
			// Delete the target mesh filter's mesh
			MeshFilter targetMeshFilter = targetGO.GetComponent<MeshFilter>();
			if (targetMeshFilter != null)
			{
				Mesh targetMesh = targetMeshFilter.sharedMesh;
				if (targetMesh != null)
				{
					if (!bDontDeletePersistantResources || !HEU_EditorUtility.IsPersistant(targetMesh))
					{
						HEU_GeneralUtility.DestroyImmediate(targetMesh);
					}

					targetMesh = null;
					targetMeshFilter.sharedMesh = null;
				}
			}

			// Delete the target mesh collider's mesh
			MeshCollider targetMeshCollider = targetGO.GetComponent<MeshCollider>();
			if (targetMeshCollider != null)
			{
				Mesh targetColliderMesh = targetMeshCollider != null ? targetMeshCollider.sharedMesh : null;
				if (targetColliderMesh != null)
				{
					if (!bDontDeletePersistantResources || !HEU_EditorUtility.IsPersistant(targetColliderMesh))
					{
						HEU_GeneralUtility.DestroyImmediate(targetColliderMesh);
					}

					targetColliderMesh = null;
					targetMeshCollider.sharedMesh = null;
				}
			}

			// Delete existing materials and textures
			MeshRenderer targetMeshRenderer = targetGO.GetComponent<MeshRenderer>();
			if (targetMeshRenderer != null && !bDontDeletePersistantResources)
			{
				Material[] targetMaterials = targetMeshRenderer.sharedMaterials;

				if (targetMaterials != null)
				{
					for (int i = 0; i < targetMaterials.Length; ++i)
					{
						Material material = targetMaterials[i];
						if (material == null)
						{
							continue;
						}

						// Diffuse texture
						if (material.HasProperty("_MainTex"))
						{
							Texture srcDiffuseTexture = material.mainTexture;
							if (srcDiffuseTexture != null)
							{
								HEU_AssetDatabase.DeleteAssetIfInBakedFolder(srcDiffuseTexture);
							}
						}

						// Normal map
						if (material.HasProperty(HEU_Defines.UNITY_SHADER_BUMP_MAP))
						{
							Texture srcNormalMap = material.GetTexture(HEU_Defines.UNITY_SHADER_BUMP_MAP);
							if (srcNormalMap != null)
							{
								HEU_AssetDatabase.DeleteAssetIfInBakedFolder(srcNormalMap);
							}
						}

						// Material
						HEU_AssetDatabase.DeleteAssetIfInBakedFolder(targetMaterials[i]);
						targetMaterials[i] = null;
					}

					targetMeshRenderer.sharedMaterials = targetMaterials;
				}
			}
		}

		/// <summary>
		/// Processs and build the mesh for this part.
		/// </summary>
		/// <param name="session">Active session to use.</param>
		/// <param name="partInfo">HAPI_PartInfo for this part.</param>
		/// <param name="bGenerateUVs">Whether to generate UVs manually.</param>
		/// <param name="bGenerateTangents">Whether to generate tangents manually.</param>
		/// <param name="bGeoIntermediate"></param>
		/// <returns>True if successfully built the mesh.</returns>
		public bool ProcessMeshPart(HEU_SessionBase session, ref HAPI_PartInfo partInfo, bool bGenerateUVs, bool bGenerateTangents, bool bGeoIntermediate)
		{
			bool bResult = true;

			if (partInfo.vertexCount > 0)
			{
				// Get the geometry and material information
				HEU_GenerateGeoCache geoCache = HEU_GenerateGeoCache.GetPopulatedGeoCache(session, _geoID, _partID);
				if(geoCache == null)
				{
					// Failed to get necessary info for generating geometry.
					return false;
				}

				// We'll remove mesh components on the gameobject, and allow GenerateMesh to re-add them as needed
				// The assumption here is that we added those on last cook, so get rid of them to have current cook re-add them.
				HEU_PartData.DestroyExistingGeneratedComponentsMeshData(_gameObject, true);
				HEU_GeneralUtility.DestroyGeneratedComponents(_gameObject);

				// Build the meshes
				bResult = GenerateMeshUsingGeoCache(session, ParentAsset, _gameObject, geoCache, bGenerateUVs, bGenerateTangents, bGeoIntermediate, IsPartInstanced());
			}

			// Always returning true for meshes without geometry as they could be object instancers
			// which are handled after all parts are created.
			return bResult;
		}

		/// <summary>
		/// Process and build the volume for this part.
		/// Only terrain (heightfield) volumes are handlded.
		/// </summary>
		/// <param name="session">Active session to use.</param>
		/// <param name="volumeInfo">Volume info for this part.</param>
		/// <returns>True if successfully built the volume.</returns>
		public bool ProcessVolumePart(HEU_SessionBase session, ref HAPI_VolumeInfo volumeInfo)
		{
			Debug.Assert(_partType == HAPI_PartType.HAPI_PARTTYPE_VOLUME, "Expected volume part type!");

			// Only want the height layer
			string volumeName = HEU_SessionManager.GetString(volumeInfo.nameSH, session);
			if (!volumeName.Equals("height"))
			{
				return false;
			}

			if(volumeInfo.zLength == 1 && volumeInfo.tupleSize == 1)
			{
				//Debug.Log("Processing HeightField part!");

				// Heightfields will be converted to terrain in Unity
				// In Houdini: size(x,y) / grid spacing => In Unity: volumeInfo.xLength, volumeInfo.yLength
				int numPoints = volumeInfo.xLength * volumeInfo.yLength;

				// Use the volumeInfo.transform to get the actual heightfield size using the scale.
				Matrix4x4 volumeTransformMatrix = HEU_HAPIUtility.GetMatrixFromHAPITransform(ref volumeInfo.transform, false);
				Vector3 position = HEU_HAPIUtility.GetPosition(ref volumeTransformMatrix);
				Vector3 scale = HEU_HAPIUtility.GetScale(ref volumeTransformMatrix);
				//Debug.LogFormat("HeightField Pos:{0}, Scale:{1}", position, scale);

				float scaledSizeX = volumeInfo.xLength * scale.x * 2f;
				float scaledSizeZ = volumeInfo.yLength * scale.z * 2f;
				//Debug.LogFormat("HeightField Scaled Size x:{0}, y:{1}", scaledSizeX, scaledSizeZ);

				//Debug.LogFormat("HeightField tileSize:{0}, xLength:{1}, yLength:{2}", volumeInfo.tileSize, volumeInfo.xLength, volumeInfo.yLength);
				//Debug.LogFormat("HeightField minX={0}, minY={1}, minZ={2}", volumeInfo.minX, volumeInfo.minY, volumeInfo.minZ);

				const int UNITY_MINIMUM_HEIGHTMAP_RESOLUTION = 33;
				if(scaledSizeX < UNITY_MINIMUM_HEIGHTMAP_RESOLUTION || scaledSizeZ < UNITY_MINIMUM_HEIGHTMAP_RESOLUTION)
				{
					Debug.LogWarningFormat("Unity Terrain has a minimum heightmap resolution of {0}. This HDA heightmap size is {1}x{2}.\nPlease resize the terrain to a value higher than this.", 
						UNITY_MINIMUM_HEIGHTMAP_RESOLUTION, scaledSizeX, scaledSizeZ);
					return false;
				}

				// Unity requires power of 2 plus 1 for heightmap resolution, and square size.

				int validSizeX = Mathf.RoundToInt(volumeInfo.xLength);
				int validSizeY = Mathf.RoundToInt(volumeInfo.yLength);

				// Use biggest dimension for square
				int squareSize = validSizeX >= validSizeY ? validSizeX : validSizeY;
				squareSize = squareSize * 2;
				
				// Get next power of two
				if(!Mathf.IsPowerOfTwo(squareSize - 1))
				{
					squareSize = Mathf.NextPowerOfTwo(squareSize - 1) + 1;
				}

				// Get the height values from Houdini and find the min and max height range.
				float[] heightValues = new float[numPoints];
				bool bResult = session.GetHeightFieldData(_geoID, _partID, heightValues, 0, numPoints);
				if(!bResult)
				{
					return false;
				}

				float minHeight = heightValues[0];
				float maxHeight = minHeight;
				for(int i = 0; i < numPoints; ++i)
				{
					float f = heightValues[i];
					if (f > maxHeight)
					{
						maxHeight = f;
					}
					else if (f < minHeight)
					{
						minHeight = f;
					}
				}

				const int UNITY_MAX_HEIGHT_RANGE = 65536;
				float heightRange = (maxHeight - minHeight);
				if (Mathf.RoundToInt(heightRange) > UNITY_MAX_HEIGHT_RANGE)
				{
					Debug.LogWarningFormat("Unity Terrain has maximum height range of {0}. This HDA height range is {1}, so it will be maxed out at {0}.\nPlease resize to within valid range!",
						UNITY_MAX_HEIGHT_RANGE, Mathf.RoundToInt(heightRange));
					heightRange = UNITY_MAX_HEIGHT_RANGE;
				}

				//Debug.LogFormat("HeightField height min={0}, max={1}, range={2}", minHeight, maxHeight, heightRange);

				int mapWidth = volumeInfo.xLength;
				int mapHeight = volumeInfo.yLength;

				int paddingWidth = squareSize - mapWidth;
				int paddingLeft = Mathf.CeilToInt(paddingWidth * 0.5f);
				int paddingRight = squareSize - paddingLeft;
				//Debug.LogFormat("Padding: Width={0}, Left={1}, Right={2}", paddingWidth, paddingLeft, paddingRight);

				int paddingHeight = squareSize - mapHeight;
				int paddingTop = Mathf.CeilToInt(paddingHeight * 0.5f);
				int paddingBottom = squareSize - paddingTop;
				//Debug.LogFormat("Padding: Height={0}, Top={1}, Bottom={2}", paddingHeight, paddingTop, paddingBottom);

				// Set height values at centre of the terrain, with padding on the sides if we resized
				float[,] unityHeights = new float[squareSize, squareSize];
				for (int y = 0; y < squareSize; ++y)
				{
					for (int x = 0; x < squareSize; ++x)
					{
						if (y >= paddingTop && y < (paddingBottom) && x >= paddingLeft && x < (paddingRight))
						{
							int ay = x - paddingLeft;
							int ax = y - paddingTop;

							// Unity expects normalized height values
							float h = heightValues[ay + ax * mapWidth] - minHeight;
							float f = h / heightRange;

							// Flip for right-hand to left-handed coordinate system
							int ix = x;
							int iy = squareSize - (y + 1);

							// Unity expects height array indexing to be [y, x].
							unityHeights[ix, iy] = f;
						}
					}
				}

				TerrainData terrainData = new TerrainData();

				// Heightmap resolution must be square power-of-two plus 1
				terrainData.heightmapResolution = squareSize;

				// TODO: Pull these from plugin settings perhaps?
				terrainData.baseMapResolution = 1024;
				terrainData.alphamapResolution = 1024;

				int detailResolution = squareSize - 1;
				// 128 is the maximum for resolutionPerPatch
				const int resolutionPerPatch = 128;
				terrainData.SetDetailResolution(detailResolution, resolutionPerPatch);

				// Note SetHeights must be called before setting size in next line, as otherwise
				// the internal terrain size will not change after setting the size.
				terrainData.SetHeights(0, 0, unityHeights);

				//Debug.Log("SquareSize: " + squareSize);
				int terrainSizeX = Mathf.RoundToInt((squareSize - 1) * scale.x * 2f);
				int terrainSizeZ = Mathf.RoundToInt((squareSize - 1) * scale.z * 2f);
				//Debug.LogFormat("Terrain size!!!!: {0}x{1}", terrainSizeX, terrainSizeZ);
				terrainData.size = new Vector3(terrainSizeX, heightRange, terrainSizeZ);

				if (squareSize != validSizeX || squareSize != validSizeY)
				{
					Debug.LogWarningFormat("Terrain will be resized from {0}x{1} to {2}x{2}. Unity requires power-of-two plus 1 square-sized heightmaps (e.g. 257x257).", scaledSizeX, scaledSizeZ, Mathf.RoundToInt(terrainSizeX));
				}

				Terrain terrain = HEU_GeneralUtility.GetOrCreateComponent<Terrain>(_gameObject);
				TerrainCollider collider = HEU_GeneralUtility.GetOrCreateComponent<TerrainCollider>(_gameObject);
				terrain.terrainData = terrainData;
				collider.terrainData = terrainData;

				// Save out terrain data to our asset cache
				// TODO: copy terrain data on bake out
				// TODO: destroy terrain data when part destroy
				ParentAsset.AddToAssetDBCache(string.Format("Part_{0}_TerrainData", PartID), terrainData, ref _assetDBTerrainData);

				terrain.Flush();

				// Unity Terrain has origin at bottom left, whereas
				// Houdini uses centre of terrain. We'll need to
				// move the terrain half way back in X and half way up in Z
				//float xmid = terrainSizeX * 0.5f;
				//float zmid = terrainSizeZ * 0.5f;

				// TODO: revisit to properly get volume position offset when splitting tiles
				float xmin, xmax, zmin, zmax, ymin, ymax, xcenter, ycenter, zcenter;
				HEU_HAPIImports.HAPI_GetVolumeBounds(ref session.GetSessionData()._HAPISession, _geoID, _partID, out xmin, out ymin, out zmin, out xmax, out ymax, out zmax, out xcenter,
					out ycenter, out zcenter);
				//Debug.LogFormat("xmin: {0}, xmax: {1}, ymin: {2}, ymax: {3}, zmin: {4}, zmax: {5}, xc: {6}, yc: {7}, zc: {8}",
				//	xmin, xmax, ymin, ymax, zmin, zmax, xcenter, ycenter, zcenter);
				_volumePosition = new Vector3(-position.x + (xcenter * 2.0f), minHeight + position.y, position.z);

				//_volumePosition = new Vector3(-position.x, minHeight + position.y, position.z);

				return true;
			}
			else
			{
				Debug.LogWarning("Non-heightfield volume type not supported!");
			}

			return false;
		}

		public bool ProcessCurvePart(HEU_SessionBase session, ref HAPI_PartInfo partInfo)
		{
			HEU_HoudiniAsset parentAsset = ParentAsset;

			if(_curve == null)
			{
				string partName = HEU_SessionManager.GetString(partInfo.nameSH, session);
				_curve = HEU_Curve.CreateSetupCurve(parentAsset, _geoNode.Editable, partName, _geoID, false);
			}
			else
			{
				_curve.UploadParameterPreset(session, _geoID, parentAsset);
			}

			_curve.SyncFromParameters(session, parentAsset);

			bool bResult = _curve.UpdateCurve(session, _partID);
			if(bResult)
			{
				_curve.GenerateMesh(_gameObject);
			}

			return bResult;
		}

		public void SyncAttributesStore(HEU_SessionBase session, HAPI_NodeId geoID, ref HAPI_PartInfo partInfo)
		{
			if(_attributesStore == null)
			{
				_attributesStore = ScriptableObject.CreateInstance<HEU_AttributesStore>();
			}

			HEU_HoudiniAsset parentAsset = ParentAsset;
			if (parentAsset != null)
			{
				_attributesStore.SyncAllAttributesFrom(session, parentAsset, geoID, ref partInfo, _gameObject);

				parentAsset.AddAttributeStore(_attributesStore);
			}
		}

		public void DestroyAttributesStore()
		{
			if(_attributesStore != null)
			{
				HEU_HoudiniAsset parentAsset = ParentAsset;
				if (parentAsset != null)
				{
					parentAsset.RemoveAttributeStore(_attributesStore);

					_attributesStore.DestroyAllData(parentAsset);
				}

				HEU_GeneralUtility.DestroyImmediate(_attributesStore);
				_attributesStore = null;
			}
		}

		/// <summary>
		/// Fill in the objInstanceInfos list with the HEU_ObjectInstanceInfos used by this part.
		/// </summary>
		/// <param name="objInstanceInfos">List to fill in</param>
		public void PopulateObjectInstanceInfos(List<HEU_ObjectInstanceInfo> objInstanceInfos)
		{
			objInstanceInfos.AddRange(_objectInstanceInfos);
		}

		/// <summary>
		/// Set object instance infos from the given part into this.
		/// </summary>
		/// <param name="otherPart"></param>
		public void SetObjectInstanceInfos(List<HEU_ObjectInstanceInfo> sourceObjectInstanceInfos)
		{
			int numSourceInfos = sourceObjectInstanceInfos.Count;
			for (int i = 0; i < numSourceInfos; ++i)
			{
				sourceObjectInstanceInfos[i]._instances.Clear();
				sourceObjectInstanceInfos[i]._partTarget = this;

				_objectInstanceInfos.Add(sourceObjectInstanceInfos[i]);
			}
		}

		/// <summary>
		/// Return list of HEU_ObjectInstanceInfo used by this part.
		/// </summary>
		/// <returns></returns>
		public List<HEU_ObjectInstanceInfo> GetObjectInstanceInfos()
		{
			return _objectInstanceInfos;
		}

		/// <summary>
		/// Helper to create a HEU_ObjectInstanceInfo, representing an instanced object
		/// containing list of instances.
		/// Adds this new object to _objectInstanceInfos.
		/// </summary>
		/// <param name="instancedObject">The source instanced object</param>
		/// <param name="instancedObjectNodeID">If instancedObject is a Houdini Engine object node, then this would be its node ID</param>
		/// <param name="instancedObjectPath">Path in Unity to the instanced object (could be empty or null if not a Unity instanced object)</param>
		/// <returns>The created object</returns>
		private HEU_ObjectInstanceInfo CreateObjectInstanceInfo(GameObject instancedObject, HAPI_NodeId instancedObjectNodeID, string instancedObjectPath)
		{
			HEU_ObjectInstanceInfo newInfo = ScriptableObject.CreateInstance<HEU_ObjectInstanceInfo>();
			newInfo._partTarget = this;
			newInfo._instancedObjectNodeID = instancedObjectNodeID;
			newInfo._instancedObjectPath = instancedObjectPath;

			HEU_InstancedInput input = new HEU_InstancedInput();
			input._instancedGameObject = instancedObject;
			newInfo._instancedInputs.Add(input);

			_objectInstanceInfos.Add(newInfo);
			return newInfo;
		}

		/// <summary>
		/// Returns HEU_ObjectInstanceInfo with matching _instancedObjectPath.
		/// </summary>
		/// <param name="path">The path to match with _instancedObjectPath</param>
		/// <returns>HEU_ObjectInstanceInfo with matching _instancedObjectPath or null if none found</returns>
		public HEU_ObjectInstanceInfo GetObjectInstanceInfoWithObjectPath(string path)
		{
			int numSourceInfos = _objectInstanceInfos.Count;
			for (int i = 0; i < numSourceInfos; ++i)
			{
				if(_objectInstanceInfos[i]._instancedObjectPath.Equals(path))
				{
					return _objectInstanceInfos[i];
				}
			}
			return null;
		}

		/// <summary>
		/// Returns HEU_ObjectInstanceInfo with matching objNodeID
		/// </summary>
		/// <param name="objNodeID">The Houdini Engine node ID to match</param>
		/// <returns>HEU_ObjectInstanceInfo with matching objNodeID or null if none found</returns>
		public HEU_ObjectInstanceInfo GetObjectInstanceInfoWithObjectID(HAPI_NodeId objNodeID)
		{
			int numSourceInfos = _objectInstanceInfos.Count;
			for (int i = 0; i < numSourceInfos; ++i)
			{
				if (_objectInstanceInfos[i]._instancedObjectNodeID == objNodeID)
				{
					return _objectInstanceInfos[i];
				}
			}
			return null;
		}

		public static string AppendBakedCloneName(string name)
		{
			return name + HEU_Defines.HEU_BAKED_CLONE;
		}

		public override string ToString()
		{
			return (!string.IsNullOrEmpty(_partName) ? ("Part: " + _partName) : base.ToString());
		}

		
	}

}   // HoudiniEngineUnity
						 