﻿/*
* Copyright (c) <2018> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using System.Collections.Generic;


namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;
	using HAPI_StringHandle = System.Int32;

	/// <summary>
	/// Contains all the attributes for an editable node (part).
	/// Addtionally contains attribute-editing tools data such
	/// as temporary mesh and collider.
	/// </summary>
	public class HEU_AttributesStore : ScriptableObject
	{
		//	DATA ------------------------------------------------------------------------------------------------------

		[SerializeField]
		private HAPI_NodeId _geoID;

		public HAPI_NodeId GeoID { get { return _geoID; } }

		[SerializeField]
		private HAPI_PartId _partID;

		public HAPI_PartId PartID { get { return _partID; } }

		[SerializeField]
		private List<HEU_AttributeData> _attributeDatas = new List<HEU_AttributeData>();

		[SerializeField]
		private bool _hasColorAttribute;

		public bool HasColorAttribute() { return _hasColorAttribute; }

		[SerializeField]
		private Mesh _outputMesh;

		public Mesh OutputMesh { get { return _outputMesh; } }

		[SerializeField]
		private MeshRenderer _outputMeshRenderer;

		[SerializeField]
		private bool _outputMeshRendererInitiallyEnabled;

		[SerializeField]
		private MeshCollider _outputMeshCollider;

		public MeshCollider OutputMeshCollider { get { return _outputMeshCollider; } }

		[SerializeField]
		private bool _outputMeshColliderInitiallyEnabled;

		[SerializeField]
		private Material _originalMaterial;

		[SerializeField]
		private Material _editableMaterial;

		[SerializeField]
		private int _editableMaterialKey;

		[SerializeField]
		private Transform _outputTransform;

		public Transform OutputTransform { get { return _outputTransform; } }


		//	LOGIC -----------------------------------------------------------------------------------------------------

		public void DestroyAllData(HEU_HoudiniAsset asset)
		{
			_attributeDatas.Clear();

			if(_editableMaterial != null)
			{
				asset.RemoveMaterial(_editableMaterial);
				_editableMaterial = null;
				_editableMaterialKey = HEU_Defines.HEU_INVALID_MATERIAL;
			}
		}

		public void SyncAllAttributesFrom(HEU_SessionBase session, HEU_HoudiniAsset asset, HAPI_NodeId geoID, ref HAPI_PartInfo partInfo, GameObject outputGameObject)
		{
			_geoID = geoID;
			_partID = partInfo.id;

			_outputTransform = outputGameObject.transform;

			// Get vertex to point association for lookup
			int[] vertexList = new int[partInfo.vertexCount];
			bool bResult = HEU_GeneralUtility.GetArray2Arg(geoID, partInfo.id, session.GetVertexList, vertexList, 0, partInfo.vertexCount);
			if (!bResult)
			{
				Debug.LogErrorFormat("Failed to sync attributes. Unable to retrieve vertex information.");
				return;
			}

			int attributePointCount = partInfo.attributeCounts[(int)HAPI_AttributeOwner.HAPI_ATTROWNER_POINT];
			string[] pointAttributeNames = new string[attributePointCount];
			if(!session.GetAttributeNames(geoID, partInfo.id, HAPI_AttributeOwner.HAPI_ATTROWNER_POINT, ref pointAttributeNames, attributePointCount))
			{
				Debug.LogErrorFormat("Failed to sync attributes. Unable to retrieve attribute names.");
				return;
			}

			// Create new list of attributes. We'll move existing attributes that are still in use as we find them.
			List<HEU_AttributeData> newAttributeDatas = new List<HEU_AttributeData>();

			foreach (string pointAttributeName in pointAttributeNames)
			{
				if(string.IsNullOrEmpty(pointAttributeName) || pointAttributeName.Equals("P"))
				{
					continue;
				}

				HAPI_AttributeInfo pointAttributeInfo = new HAPI_AttributeInfo();
				if(session.GetAttributeInfo(geoID, partInfo.id, pointAttributeName, HAPI_AttributeOwner.HAPI_ATTROWNER_POINT, ref pointAttributeInfo))
				{
					HEU_AttributeData attrData = GetAttributeData(pointAttributeName);
					if (attrData == null)
					{
						// Attribute data not found. Create it.

						attrData = CreateAttribute(pointAttributeName, ref pointAttributeInfo);
						//Debug.LogFormat("Created attribute data: {0}", pointAttributeName);
					}

					// Add to new list.
					newAttributeDatas.Add(attrData);

					// Sync the attribute info to data.
					SyncAttributeFromInfo(session, geoID, ref partInfo, attrData, ref pointAttributeInfo, vertexList);

					if(pointAttributeName.Equals(HEU_Defines.HAPI_ATTRIB_COLOR) || pointAttributeInfo.typeInfo == HAPI_AttributeTypeInfo.HAPI_ATTRIBUTE_TYPE_COLOR)
					{
						_hasColorAttribute = true;
					}
				}
				else
				{
					// Failed to get point attribute info!
				}
			}

			// Overwriting the old list with the new should automatically remove unused attribute datas.
			_attributeDatas = newAttributeDatas;

			// Set mesh and material data
			if(partInfo.type == HAPI_PartType.HAPI_PARTTYPE_MESH 
				|| partInfo.type == HAPI_PartType.HAPI_PARTTYPE_BOX
				|| partInfo.type == HAPI_PartType.HAPI_PARTTYPE_SPHERE)
			{
				// Get the generated mesh. If mesh is missing, nothing we can do.
				MeshFilter meshFilter = outputGameObject.GetComponent<MeshFilter>();
				if(meshFilter != null && meshFilter.sharedMesh != null)
				{
					_outputMesh = meshFilter.sharedMesh;
				}

				// Get mesh renderer or add new one if not found.
				_outputMeshRenderer = outputGameObject.GetComponent<MeshRenderer>();
				if(_outputMeshRenderer == null)
				{
					_outputMeshRenderer = outputGameObject.AddComponent<MeshRenderer>();
					_outputMeshRenderer.enabled = false;
					_outputMeshRendererInitiallyEnabled = false;
				}
				else
				{
					_outputMeshRendererInitiallyEnabled = _outputMeshRenderer.enabled;
				}

				// Get mesh collider or add new one if not found (using MeshFilter's mesh_
				_outputMeshCollider = outputGameObject.GetComponent<MeshCollider>();
				if(_outputMeshCollider == null)
				{
					_outputMeshCollider = outputGameObject.AddComponent<MeshCollider>();
					_outputMeshCollider.sharedMesh = _outputMesh;
					_outputMeshCollider.enabled = false;
				}
				else
				{
					_outputMeshColliderInitiallyEnabled = _outputMeshCollider.enabled;
				}

				// Re-use editable material if exists, or copy it from mesh (original material) / create new one just for editing
				if (_editableMaterial == null)
				{
					_originalMaterial = _outputMeshRenderer.sharedMaterial;

					HEU_MaterialData editableMaterialData = null;
					if (_originalMaterial == null)
					{
						editableMaterialData = asset.CreateMaterialInCache(HEU_Defines.EDITABLE_MATERIAL_KEY, HEU_Defines.EDITABLE_MATERIAL, HEU_MaterialData.Source.HOUDINI, true);
					}
					else
					{
						// Copy the original material, and force it to use the vertex shader
						editableMaterialData = asset.CreateMaterialCopyInCache(_originalMaterial, HEU_Defines.EDITABLE_MATERIAL_KEY, HEU_Defines.EDITABLE_MATERIAL);
						editableMaterialData._material.shader = HEU_MaterialFactory.FindPluginShader(HEU_Defines.DEFAULT_VERTEXCOLOR_SHADER);
					}

					if (editableMaterialData != null)
					{
						_editableMaterial = editableMaterialData._material;
						_editableMaterialKey = editableMaterialData._materialKey;
					}
				}
				else
				{
					// Get the original material as long as its not the editable material
					if (_editableMaterial != _outputMeshRenderer.sharedMaterial)
					{
						_originalMaterial = _outputMeshRenderer.sharedMaterial;
					}
				}
			}
		}

		public void SyncAttributeFromInfo(HEU_SessionBase session, HAPI_NodeId geoID, ref HAPI_PartInfo partInfo, HEU_AttributeData attributeData, ref HAPI_AttributeInfo attributeInfo, int[] vertexList)
		{
			//Debug.LogFormat("Syncing attribute data: {0}", attributeData._name);

			HAPI_AttributeInfo previousAttributeInfo = attributeData._attributeInfo;

			attributeData._attributeInfo = attributeInfo;

			int tupleSize = attributeInfo.tupleSize;
			int vertexCount = partInfo.vertexCount;
			int arraySize = vertexCount * tupleSize;

			// First reset arrays if the type had changed since last sync
			if ((attributeInfo.storage == HAPI_StorageType.HAPI_STORAGETYPE_INT && attributeData._attributeType != HEU_AttributeData.AttributeType.INT) ||
				(attributeInfo.storage == HAPI_StorageType.HAPI_STORAGETYPE_FLOAT && attributeData._attributeType != HEU_AttributeData.AttributeType.FLOAT) ||
				(attributeInfo.storage == HAPI_StorageType.HAPI_STORAGETYPE_STRING && attributeData._attributeType != HEU_AttributeData.AttributeType.STRING))
			{
				// Type changed, so reset arrays
				attributeData._floatValues = null;
				attributeData._stringValues = null;
				attributeData._intValues = null;

				attributeData._attributeState = HEU_AttributeData.AttributeState.INVALID;
			}

			// Make sure the internal array is correctly sized for syncing.
			if (attributeData._attributeType == HEU_AttributeData.AttributeType.INT)
			{
				if(attributeData._intValues == null)
				{
					attributeData._intValues = new int[arraySize];
				}
				else if(attributeData._intValues.Length != arraySize)
				{
					System.Array.Resize<int>(ref attributeData._intValues, arraySize);
				}
				attributeData._floatValues = null;
				attributeData._stringValues = null;
			}
			else if (attributeData._attributeType == HEU_AttributeData.AttributeType.FLOAT)
			{
				if (attributeData._floatValues == null)
				{
					attributeData._floatValues = new float[arraySize];
				}
				else if (attributeData._floatValues.Length != arraySize)
				{
					System.Array.Resize<float>(ref attributeData._floatValues, arraySize);
				}
				attributeData._intValues = null;
				attributeData._stringValues = null;
			}
			else if (attributeData._attributeType == HEU_AttributeData.AttributeType.STRING)
			{
				if (attributeData._stringValues == null)
				{
					attributeData._stringValues = new string[arraySize];
				}
				else if (attributeData._stringValues.Length != arraySize)
				{
					System.Array.Resize<string>(ref attributeData._stringValues, arraySize);
				}
				attributeData._intValues = null;
				attributeData._floatValues = null;
			}

			// Now sync. If _attributeState is INVALID, we get from Houdini. If its LOCAL_DIRTY, we send to Houdini.
			if (attributeData._attributeType == HEU_AttributeData.AttributeType.INT)
			{
				if (attributeData._attributeState == HEU_AttributeData.AttributeState.INVALID)
				{
					int[] data = new int[0];
					HEU_GeneralUtility.GetAttribute(session, geoID, partInfo.id, attributeData._name, ref attributeInfo, ref data, session.GetAttributeIntData);
					for (int i = 0; i < vertexCount; ++i)
					{
						for (int tuple = 0; tuple < tupleSize; ++tuple)
						{
							attributeData._intValues[i * tupleSize + tuple] = data[vertexList[i] * tupleSize + tuple];
						}
					}
				}
			}
			else if (attributeData._attributeType == HEU_AttributeData.AttributeType.FLOAT)
			{
				if (attributeData._attributeState == HEU_AttributeData.AttributeState.INVALID)
				{
					float[] data = new float[0];
					HEU_GeneralUtility.GetAttribute(session, geoID, partInfo.id, attributeData._name, ref attributeInfo, ref data, session.GetAttributeFloatData);
					for (int i = 0; i < vertexCount; ++i)
					{
						for (int tuple = 0; tuple < tupleSize; ++tuple)
						{
							attributeData._floatValues[i * tupleSize + tuple] = data[vertexList[i] * tupleSize + tuple];
						}
					}
				}
			}
			else if (attributeData._attributeType == HEU_AttributeData.AttributeType.STRING)
			{
				if (attributeData._attributeState == HEU_AttributeData.AttributeState.INVALID)
				{
					HAPI_StringHandle[] data = new HAPI_StringHandle[0];
					HEU_GeneralUtility.GetAttribute(session, geoID, partInfo.id, attributeData._name, ref attributeInfo, ref data, session.GetAttributeStringData);
					for (int i = 0; i < vertexCount; ++i)
					{
						for (int tuple = 0; tuple < tupleSize; ++tuple)
						{
							HAPI_StringHandle stringHandle = data[vertexList[i] * tupleSize + tuple];
							attributeData._stringValues[i * tupleSize + tuple] = HEU_SessionManager.GetString(stringHandle, session);
						}
					}
				}
			}

			SetAttributeDataSyncd(attributeData);
		}

		public void SyncDirtyAttributesToHoudini(HEU_SessionBase session)
		{
			HAPI_PartInfo partInfo = new HoudiniEngineUnity.HAPI_PartInfo();
			if(!session.GetPartInfo(_geoID, _partID, ref partInfo))
			{
				Debug.LogErrorFormat("Unable to get partinfo for part {0}. Can't sync attributes!", _partID);
				return;
			}

			bool bSyncd = false;
			foreach (HEU_AttributeData attrData in _attributeDatas)
			{
				if (!string.IsNullOrEmpty(attrData._name) && attrData._attributeState == HEU_AttributeData.AttributeState.LOCAL_DIRTY)
				{
					HAPI_AttributeInfo pointAttributeInfo = new HAPI_AttributeInfo();
					if (session.GetAttributeInfo(_geoID, partInfo.id, attrData._name, HAPI_AttributeOwner.HAPI_ATTROWNER_POINT, ref pointAttributeInfo))
					{
						bSyncd |= SyncAttributeToHoudini(session, _geoID, ref partInfo, attrData, ref pointAttributeInfo);
					}
				}
			}

			if(bSyncd)
			{
				// TODO: review necessity
				session.CommitGeo(_geoID);
			}
		}

		public bool SyncAttributeToHoudini(HEU_SessionBase session, HAPI_NodeId geoID, ref HAPI_PartInfo partInfo, HEU_AttributeData attributeData, ref HAPI_AttributeInfo attributeInfo)
		{
			if(attributeData._attributeState == HEU_AttributeData.AttributeState.INVALID)
			{
				Debug.LogErrorFormat("Unable to sync attribute {0} due to it being invalid!", attributeData._name);
				return false;
			}

			// Using 0 as part ID as HAPI need it
			HAPI_PartId partID = 0;

			// TODO: review necessity
			if (!session.AddAttribute(geoID, partID, attributeData._name, ref attributeInfo))
			{
				Debug.LogErrorFormat("Failed to add attribute: {0}", attributeData._name);
				return false;
			}

			// TODO: mapping to points from vertices!

			bool bResult = false;
			if (attributeData._attributeType == HEU_AttributeData.AttributeType.INT)
			{
				bResult = HEU_GeneralUtility.SetAttribute(geoID, partID, attributeData._name, ref attributeInfo, attributeData._intValues, session.SetAttributeIntData);
			}
			else if (attributeData._attributeType == HEU_AttributeData.AttributeType.FLOAT)
			{
				bResult = HEU_GeneralUtility.SetAttribute(geoID, partID, attributeData._name, ref attributeInfo, attributeData._floatValues, session.SetAttributeFloatData);
			}
			else if (attributeData._attributeType == HEU_AttributeData.AttributeType.STRING)
			{
				bResult = HEU_GeneralUtility.SetAttribute(geoID, partID, attributeData._name, ref attributeInfo, attributeData._stringValues, session.SetAttributeStringData);
			}

			if(!bResult)
			{
				Debug.LogErrorFormat("Failed to set attribute {0}!", attributeData._name);
			}

			SetAttributeDataSyncd(attributeData);

			return bResult;
		}

		private static void SetAttributeDataSyncd(HEU_AttributeData attributeData)
		{
			attributeData._attributeState = HEU_AttributeData.AttributeState.SYNCED;
		}

		private static void SetAttributeDataDirty(HEU_AttributeData attributeData)
		{
			attributeData._attributeState = HEU_AttributeData.AttributeState.LOCAL_DIRTY;
		}

		public HEU_AttributeData CreateAttribute(string attributeName, ref HAPI_AttributeInfo attributeInfo)
		{
			HEU_AttributeData.AttributeType attributeType = HEU_AttributeData.AttributeType.UNDEFINED;
			if (attributeInfo.storage == HAPI_StorageType.HAPI_STORAGETYPE_INT)
			{
				attributeType = HEU_AttributeData.AttributeType.INT;
			}
			else if (attributeInfo.storage == HAPI_StorageType.HAPI_STORAGETYPE_FLOAT)
			{
				attributeType = HEU_AttributeData.AttributeType.FLOAT;
			}
			else if (attributeInfo.storage == HAPI_StorageType.HAPI_STORAGETYPE_STRING)
			{
				attributeType = HEU_AttributeData.AttributeType.STRING;
			}

			HEU_AttributeData attributeData = new HEU_AttributeData();
			attributeData._name = attributeName;
			attributeData._attributeType = attributeType;
			attributeData._attributeInfo = attributeInfo;
			attributeData._attributeState = HEU_AttributeData.AttributeState.INVALID;

			return attributeData;
		}

		public HEU_AttributeData GetAttributeData(string name)
		{
			foreach(HEU_AttributeData attr in _attributeDatas)
			{
				if(attr._name.Equals(name))
				{
					return attr;
				}
			}
			return null;
		}

		public HEU_AttributeData GetAttributeData(int index)
		{
			if(index >= 0 && index < _attributeDatas.Count)
			{
				return _attributeDatas[index];
			}
			return null;
		}

		public List<string> GetAttributeNames()
		{
			List<string> attributeNames = new List<string>();
			foreach(HEU_AttributeData data in _attributeDatas)
			{
				attributeNames.Add(data._name);
			}
			return attributeNames;
		}

		public void EnableEditMode()
		{
			if (_outputMeshRenderer != null)
			{
				_outputMeshRenderer.sharedMaterial = _editableMaterial;
				_outputMeshRenderer.enabled = true;
			}

			if(_outputMeshCollider != null)
			{
				_outputMeshCollider.sharedMesh = _outputMesh;
				// Collider needs to be disabled then enabled to properly activate
				_outputMeshCollider.enabled = false;
				_outputMeshCollider.enabled = true;
			}
		}

		public void DisableEditMode()
		{
			if (_outputMeshRenderer != null)
			{
				_outputMeshRenderer.sharedMaterial = _originalMaterial;
				_outputMeshRenderer.enabled = false;// _outputMeshRendererInitiallyEnabled;
			}

			if (_outputMeshCollider != null)
			{
				_outputMeshCollider.sharedMesh = _outputMesh;
				_outputMeshCollider.enabled = false;// _outputMeshColliderInitiallyEnabled;
			}
		}

		public void PaintAttribute(HEU_AttributeData attributeData, HEU_ToolsInfo sourceTools, int vertexIndex, float paintFactor, SetAttributeValues setAttrFunc)
		{
			if(attributeData._attributeState == HEU_AttributeData.AttributeState.INVALID)
			{
				return;
			}

			int tupleSize = attributeData._attributeInfo.tupleSize;
			int startComponentIndex = 0;
			int endComponentIndex = tupleSize;

			for (int i = startComponentIndex; i < endComponentIndex; ++i)
			{
				int targetIndex = vertexIndex * attributeData._attributeInfo.tupleSize + i;
				setAttrFunc(attributeData, targetIndex, sourceTools, i, paintFactor);
			}

			//Debug.Log("Setting dirty!");
			SetAttributeDataDirty(attributeData);
		}

		public delegate void SetAttributeValues(HEU_AttributeData attributeData, int targetIndex, HEU_ToolsInfo sourceTools, int sourceIndex, float factor);

		public static void SetAttributeValueInt(HEU_AttributeData attributeData, int targetIndex, HEU_ToolsInfo sourceTools, int sourceIndex, float factor)
		{
			attributeData._intValues[targetIndex] = sourceTools._paintIntValue[sourceIndex];
		}

		public static void SetAttributeValueFloat(HEU_AttributeData attributeData, int targetIndex, HEU_ToolsInfo sourceTools, int sourceIndex, float factor)
		{
			attributeData._floatValues[targetIndex] = sourceTools._paintFloatValue[sourceIndex];
		}

		public static void SetAttributeValueString(HEU_AttributeData attributeData, int targetIndex, HEU_ToolsInfo sourceTools, int sourceIndex, float factor)
		{
			attributeData._stringValues[targetIndex] = sourceTools._paintStringValue;
		}

		public bool AreAttributesDirty()
		{
			foreach(HEU_AttributeData attrData in _attributeDatas)
			{
				if(attrData._attributeState == HEU_AttributeData.AttributeState.LOCAL_DIRTY)
				{
					return true;
				}
			}
			return false;
		}
	}

}   // HoudiniEngineUnity