﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Provides a programmer-centric API for querying and setting paramter values for an asset.
	/// </summary>
	public static class HEU_ParameterAccessor
	{
		public static bool GetToggle(HEU_HoudiniAsset asset, string paramName, out bool outValue)
		{
			outValue = false;
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsToggle())
			{
				outValue = paramData._toggle;
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Query failed. Asset [{0}]'s Parameter [{1}] is not a valid toggle!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetToggle(HEU_HoudiniAsset asset, string paramName, bool setValue)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsToggle())
			{
				paramData._toggle = setValue;
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid toggle!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool GetInt(HEU_HoudiniAsset asset, string paramName, out int outValue)
		{
			outValue = 0;
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if(paramData != null && paramData.IsInt())
			{
				outValue = paramData._intValues[0];
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Query failed. Asset [{0}]'s Parameter [{1}] is not a valid int!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetInt(HEU_HoudiniAsset asset, string paramName, int setValue)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsInt())
			{
				paramData._intValues[0] = setValue;
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid int!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool GetFloat(HEU_HoudiniAsset asset, string paramName, out float outValue)
		{
			outValue = 0;
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsFloat())
			{
				outValue = paramData._floatValues[0];
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Query failed. Asset [{0}]'s Parameter [{1}] is not a valid float!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool GetFloats(HEU_HoudiniAsset asset, string paramName, out float[] outValues)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsFloat())
			{
				outValues = paramData._floatValues;
				return true;
			}
			else
			{
				outValues = new float[0];
				Debug.LogWarningFormat("{0}: Query failed. Asset [{0}]'s Parameter [{1}] is not a valid float!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetFloat(HEU_HoudiniAsset asset, string paramName, float setValue)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsFloat())
			{
				paramData._floatValues[0] = setValue;
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid float!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetFloats(HEU_HoudiniAsset asset, string paramName, float[] setValues)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData.IsFloat())
			{
				paramData._floatValues = setValues;
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid float!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool GetString(HEU_HoudiniAsset asset, string paramName, out string outValue)
		{
			outValue = null;
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && (paramData.IsString() || paramData.IsPathFile()))
			{
				outValue = paramData._stringValues[0];
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Query failed. Asset [{0}]'s Parameter [{1}] is not a valid string!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetString(HEU_HoudiniAsset asset, string paramName, string setValue)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && (paramData.IsString() || paramData.IsPathFile()))
			{
				paramData._stringValues[0] = setValue;
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid string!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetChoice(HEU_HoudiniAsset asset, string paramName, int setValue)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData._parmInfo.choiceCount > 0 && setValue >= 0 && setValue < paramData._choiceIntValues.Length)
			{
				paramData._intValues[0] = paramData._choiceIntValues[setValue];
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid choice!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool GetChoice(HEU_HoudiniAsset asset, string paramName, out int outValue)
		{
			outValue = 0;
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData._parmInfo.choiceCount > 0)
			{
				outValue = paramData._intValues[0];
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Query failed. Asset [{0}]'s Parameter [{1}] is not a valid choice!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool SetInputNode(HEU_HoudiniAsset asset, string paramName, GameObject obj, int index)
		{
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData._paramInputNode != null)
			{
				if(index < paramData._paramInputNode.NumInputObjects())
				{
					paramData._paramInputNode.InsertInputObject(index, obj);
				}
				else
				{
					paramData._paramInputNode.AddInputObjectAtEnd(obj);
				}
				return true;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid input parameter!", asset.AssetName, paramName);
				return false;
			}
		}

		public static bool GetInputNode(HEU_HoudiniAsset asset, string paramName, int index, out GameObject obj)
		{
			obj = null;
			HEU_ParameterData paramData = asset.Parameters.GetParameter(paramName);
			if (paramData != null && paramData._paramInputNode != null)
			{
				obj = paramData._paramInputNode.GetInputObject(index)._gameObject;
				return obj != null;
			}
			else
			{
				Debug.LogWarningFormat("{0}: Set failed. Asset [{0}]'s Parameter [{1}] is not a valid input parameter!", asset.AssetName, paramName);
				return false;
			}
		}
	}

}   // HoudiniEngineUnity