﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_StringHandle = System.Int32;
	using HAPI_ParmId = System.Int32;

	/// <summary>
	/// Represents a parameter (HAPI_ParmInfo), with data storage for the modifiable values.
	/// Note that this is not derived from ScriptableObject due to limitation in Unity's
	/// serialization system where we are unable to access properties in a ScriptablObject
	/// from the UI (see HEU_ParametersUI::DrawParmProperty).
	/// Additionally, due to Unity's serialization not serializing child classes
	/// of parent classes that themselves are not derived from ScriptableObject,
	/// we are using this class to store all data types (int, float, string, etc).
	/// Also note that storing these in an array will create inline serialization
	/// as opposed to serializing just the references (i.e. duplicates will be 
	/// made if there 2 references to same object in a list).
	/// So don't store more than 1 references to these. Currently these only live in HEU_Parameters.
	/// If you change the name of the members here, also make sure to update the property serialization
	/// queries in HEU_ParameterUI, as they are string-based.
	/// </summary>
	[System.Serializable]
	public sealed class HEU_ParameterData
	{
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// Stored data

		// Index value on Unity side on the asset parameter list
		public int _unityIndex;

		// Cacheing these as they require a look up via HAPI_ParmInfo
		public string _name;
		public string _labelName;

		// List of childrens' indices. Used for fast look up when drawing UI.
		public List<int> _childParameterIDs = new List<int>();

		public int _choiceValue;

		// Store as array values to handle any sized parameters
		public int[] _intValues;
		public float[] _floatValues;
		public string[] _stringValues;

		// Storing toggle as bool so that the UI can display it as a toggle.
		// Tried using _intValues above, but Inspector would display as an int.
		public bool _toggle;

		public Color _color;
		public Gradient _gradient;
		public AnimationCurve _animCurve;

		// Choices
		public GUIContent[] _choiceLabels;
		public string[] _choiceStringValues;
		public int[] _choiceIntValues;

		// Cacheing the HAPI_ParmInfo allows us to query meta data that only changes on Houdini side.
		public HAPI_ParmInfo _parmInfo;

		// Editor UI specific
		public bool _showChildren = false;

		// Cache
		public string _fileTypeInfo;

		// Folder list children processed
		public int _folderListChildrenProcessed;

		// Folder list tab selected index
		public int _tabSelectedIndex;

		// Input node info
		public HEU_InputNode _paramInputNode;


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// Properties & Functions

		public HAPI_ParmId ParmID { get { return _parmInfo.id; } }
		public HAPI_ParmId ParentID { get { return _parmInfo.parentId; } }

		public int ChildIndex { get { return _parmInfo.childIndex; } }
		public int ParmSize { get { return _parmInfo.size; } }

		public bool IsInt() { return _parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_INT; }
		public bool IsFloat() { return _parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_FLOAT; }
		public bool IsString() { return _parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_STRING; }

		public bool IsPathFile() { return (_parmInfo.type >= HAPI_ParmType.HAPI_PARMTYPE_PATH_START 
				&& _parmInfo.type <= HAPI_ParmType.HAPI_PARMTYPE_PATH_END); }

		public bool HasMin() { return _parmInfo.hasMin; }
		public bool HasMax() { return _parmInfo.hasMax; }
		public bool HasUIMin() { return _parmInfo.hasUIMin; }
		public bool HasUIMax() { return _parmInfo.hasUIMax; }

		public int IntMin { get { return Mathf.RoundToInt(_parmInfo.min); } }
		public int IntMax { get { return Mathf.RoundToInt(_parmInfo.max); } }

		public int IntUIMin { get { return Mathf.RoundToInt(_parmInfo.UIMin); } }
		public int IntUIMax { get { return Mathf.RoundToInt(_parmInfo.UIMax); } }

		public float FloatMin { get { return _parmInfo.min; } }
		public float FloatMax { get { return _parmInfo.max; } }

		public float FloatUIMin { get { return _parmInfo.UIMin; } }
		public float FloatUIMax { get { return _parmInfo.UIMax; } }

		public bool IsContainer()
		{
			return (_childParameterIDs != null) ? (_childParameterIDs.Count > 0) : false;
		}

		public bool IsMultiParam()
		{
			return _parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_MULTIPARMLIST;
		}

		public bool IsRamp()
		{
			return (_parmInfo.rampType > HAPI_RampType.HAPI_RAMPTYPE_INVALID
						&& _parmInfo.rampType < HAPI_RampType.HAPI_RAMPTYPE_MAX);
		}

		public bool IsToggle()
		{
			return (_parmInfo.type == HAPI_ParmType.HAPI_PARMTYPE_TOGGLE);
		}
	}

}   // HoudiniEngineUnity