﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_ParmId = System.Int32;


	/// <summary>
	/// Contains utility functions for working with parameters
	/// </summary>
	public static class HEU_ParameterUtility
	{

		public static int GetParameterIndexFromName(HEU_SessionBase session, HAPI_ParmInfo[] parameters, string parameterName)
		{
			if(parameters != null && parameters.Length > 0)
			{
				int numParameters = parameters.Length;
				for(int i = 0; i < numParameters; ++i)
				{
					if(HEU_SessionManager.GetString(parameters[i].nameSH, session).Equals(parameterName))
					{
						return i;
					}
				}
			}
			return -1;
		}

		public static int GetParameterIndexFromNameOrTag(HEU_SessionBase session, HAPI_NodeId nodeID, HAPI_ParmInfo[] parameters, string parameterName)
		{
			int parameterIndex = GetParameterIndexFromName(session, parameters, parameterName);
			if (parameterIndex < 0)
			{
				// Try to find tag instead
				parameterIndex = HEU_Defines.HEU_INVALID_NODE_ID;
				session.GetParmWithTag(nodeID, parameterName, ref parameterIndex);
			}
			return parameterIndex;
		}

		public static float GetParameterFloatValue(HEU_SessionBase session, HAPI_NodeId nodeID, HAPI_ParmInfo[] parameters, string parameterName, float defaultValue)
		{
			int parameterIndex = GetParameterIndexFromNameOrTag(session, nodeID, parameters, parameterName);
			if(parameterIndex < 0 || parameterIndex >= parameters.Length)
			{
				return defaultValue;
			}

			int valueIndex = parameters[parameterIndex].floatValuesIndex;
			float[] value = new float[1];

			if(session.GetParamFloatValues(nodeID, value, valueIndex, 1))
			{
				return value[0];
			}

			return defaultValue;
		}

		public static Color GetParameterColor3Value(HEU_SessionBase session, HAPI_NodeId nodeID, HAPI_ParmInfo[] parameters, string parameterName, Color defaultValue)
		{
			int parameterIndex = GetParameterIndexFromNameOrTag(session, nodeID, parameters, parameterName);
			if (parameterIndex < 0 || parameterIndex >= parameters.Length)
			{
				return defaultValue;
			}

			if(parameters[parameterIndex].size < 3)
			{
				Debug.LogError("Parameter size not large enough to be a Color3");
				return defaultValue;
			}

			int valueIndex = parameters[parameterIndex].floatValuesIndex;
			float[] value = new float[3];

			if (session.GetParamFloatValues(nodeID, value, valueIndex, 3))
			{
				return new Color(value[0], value[1], value[2], 1f);
			}
			return defaultValue;
		}
	}

}   // HoudiniEngineUnity