﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/


using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Base class for all Houdini Engine Unity tasks.
	/// </summary>
	public abstract class HEU_Task
	{
		public enum TaskStatus
		{
			NONE,
			PENDING_START,
			STARTED,
			REQUIRE_UPDATE,
			PENDING_COMPLETE,
			COMPLETED,
			UNUSED
		}
		public TaskStatus _status;

		public enum TaskResult
		{
			NONE,
			SUCCESS,
			FAILED,
			KILLED
		}
		public TaskResult _result;

		private System.Guid _guid;
		public System.Guid TaskGuid { get { return _guid; } }

		public HEU_Task()
		{
			_guid = System.Guid.NewGuid();
		}

		public delegate void TaskCallback(HEU_Task task);

		public TaskCallback _taskCompletedDelegate;

		public abstract void DoTask();

		public virtual void UpdateTask() { }

		public abstract void KillTask();

		public abstract void CompleteTask(TaskResult result);
	}


	/// <summary>
	/// Asset-specific class for Houdini Engine Unity tasks.
	/// </summary>
	public class HEU_AssetTask : HEU_Task
	{
		public enum BuildType
		{
			NONE,
			LOAD,
			COOK,
			RELOAD
		}
		public BuildType _buildType;

		public HEU_HoudiniAsset _asset;

		public string _assetPath;

		public Vector3 _position = Vector3.zero;

		public bool _buildResult;

		public long _forceSessionID = HEU_SessionData.INVALID_SESSION_ID;

		public HEU_SessionBase GetTaskSession()
		{
			if(_forceSessionID == HEU_SessionData.INVALID_SESSION_ID)
			{
				return HEU_SessionManager.GetOrCreateDefaultSession();
			}
			else
			{
				return HEU_SessionManager.GetSessionWithID(_forceSessionID);
			}
		}


		public override void DoTask()
		{
			if(_buildType == BuildType.LOAD)
			{
				if (string.IsNullOrEmpty(_assetPath))
				{
					// Bad path so fail		
					HEU_TaskManager.CompleteTask(this, TaskResult.FAILED);
				}
				else
				{
					// File-based HDA
					GameObject newGO = HEU_HAPIUtility.InstantiateHDA(_assetPath, _position, GetTaskSession());
					if(newGO != null && newGO.GetComponent<HEU_HoudiniAssetRoot>() != null)
					{
						// Add to post-load callback
						_asset = newGO.GetComponent<HEU_HoudiniAssetRoot>()._houdiniAsset;
						_asset._reloadEvent.AddListener(CookCompletedCallback);
					}
					else
					{
						HEU_TaskManager.CompleteTask(this, TaskResult.FAILED);
					}
				}
			}
			else if(_buildType == BuildType.COOK)
			{
				_asset._cookedEvent.RemoveListener(CookCompletedCallback);
				_asset._cookedEvent.AddListener(CookCompletedCallback);
				_asset.RequestCook(false, true, false);
			}
			else if(_buildType == BuildType.RELOAD)
			{
				_asset._reloadEvent.RemoveListener(CookCompletedCallback);
				_asset._reloadEvent.AddListener(CookCompletedCallback);
				_asset.RequestReload(true);
			}
		}

		public override void KillTask()
		{
			if(_asset != null)
			{
				_asset._reloadEvent.RemoveListener(CookCompletedCallback);
				_asset._cookedEvent.RemoveListener(CookCompletedCallback);
			}
		}

		public override void CompleteTask(TaskResult result)
		{
			if (_asset != null)
			{
				_asset._reloadEvent.RemoveListener(CookCompletedCallback);
				_asset._cookedEvent.RemoveListener(CookCompletedCallback);
			}
		}

		private void CookCompletedCallback(HEU_HoudiniAsset asset, bool bSuccess, List<GameObject> outputs)
		{
			if (_status == HEU_Task.TaskStatus.STARTED)
			{
				HEU_TaskManager.CompleteTask(this, bSuccess ? TaskResult.SUCCESS : TaskResult.FAILED);
			}
		}
	}

}   // HoudiniEngineUnity