﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;

#if UNITY_EDITOR
using UnityEditor;
#endif

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Wrapper around Unity Editor functions.
	/// </summary>
	public static class HEU_EditorUtility
	{
		/// <summary>
		/// Helper to mark current scene dirty so that Unity's save system will save out any procedural changes.
		/// </summary>
		public static void MarkSceneDirty()
		{
#if UNITY_EDITOR
			if (Application.isEditor && !Application.isPlaying)
			{
				UnityEditor.SceneManagement.EditorSceneManager.MarkSceneDirty(UnityEditor.SceneManagement.EditorSceneManager.GetActiveScene());
			}
#endif
		}

		public static void SelectObject(GameObject gameObject)
		{
#if UNITY_EDITOR
			Selection.objects = new GameObject[] { gameObject };
#endif
		}

		public static void SelectObjects(GameObject[] gameObjects)
		{
#if UNITY_EDITOR
			Selection.objects = gameObjects;
#endif
		}

		public static GameObject CreatePrefab(string path, GameObject go)
		{
#if UNITY_EDITOR
			return PrefabUtility.CreatePrefab(path, go);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		public static bool IsEditorPlaying()
		{
#if UNITY_EDITOR
			return EditorApplication.isPlaying;
#else
			return false;
#endif
		}

		public enum HEU_ReplacePrefabOptions
		{
			// Replaces prefabs by matching pre-existing connections to the prefab.
			Default = 0,

			// Connects the passed objects to the prefab after uploading the prefab.
			ConnectToPrefab = 1,

			// Replaces the prefab using name based lookup in the transform hierarchy.
			ReplaceNameBased = 2
		}

		public static GameObject ReplacePrefab(GameObject go, Object targetPrefab, HEU_ReplacePrefabOptions heuOptions)
		{
#if UNITY_EDITOR
			ReplacePrefabOptions unityOptions = ReplacePrefabOptions.Default;
			switch(heuOptions)
			{
				case HEU_ReplacePrefabOptions.Default:			unityOptions = ReplacePrefabOptions.Default; break;
				case HEU_ReplacePrefabOptions.ConnectToPrefab:	unityOptions = ReplacePrefabOptions.ConnectToPrefab; break;
				case HEU_ReplacePrefabOptions.ReplaceNameBased: unityOptions = ReplacePrefabOptions.ReplaceNameBased; break;
				default: Debug.LogFormat("Unsupported replace prefab option: {0}", heuOptions); break;
			}

			return PrefabUtility.ReplacePrefab(go, targetPrefab, unityOptions);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		/// <summary>
		/// Returns true if given GameObject is an instance of a prefab.
		/// </summary>
		/// <param name="go">GameObject to check</param>
		/// <returns>True if given GameObject is an instance of a prefab</returns>
		public static bool IsPrefabInstance(GameObject go)
		{
#if UNITY_EDITOR
			return PrefabUtility.GetPrefabParent(go) != null && PrefabUtility.GetPrefabObject(go) != null;
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return false;
#endif
		}

		/// <summary>
		/// Returns true if given GameObject is a prefab (and not an instance of a prefab).
		/// </summary>
		/// <param name="go">GameObject to check</param>
		/// <returns>True if given GameObject is a prefab</returns>
		public static bool IsPrefabOriginal(GameObject go)
		{
#if UNITY_EDITOR
			return PrefabUtility.GetPrefabParent(go) == null && PrefabUtility.GetPrefabObject(go) != null;
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return false;
#endif
		}

		/// <summary>
		/// Returns true if given GameObject is a disconnected instance of a prefab.
		/// </summary>
		/// <param name="go">GameObject to check</param>
		/// <returns>True if given GameObject is a disconnected instance of a prefab</returns>
		public static bool IsDisconnectedPrefabInstance(GameObject go)
		{
#if UNITY_EDITOR
			return PrefabUtility.GetPrefabParent(go) != null && PrefabUtility.GetPrefabObject(go) == null;
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return false;
#endif
		}

		public static Object GetPrefabParent(GameObject go)
		{
#if UNITY_EDITOR
			return PrefabUtility.GetPrefabParent(go);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		public static GameObject ConnectGameObjectToPrefab(GameObject go, GameObject sourcePrefab)
		{
#if UNITY_EDITOR
			return PrefabUtility.ConnectGameObjectToPrefab(go, sourcePrefab);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		public static Object InstantiatePrefab(GameObject prefabOriginal)
		{
#if UNITY_EDITOR
			return PrefabUtility.InstantiatePrefab(prefabOriginal);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		public static GameObject InstantiateGameObject(GameObject sourceGameObject,  Transform parentTransform, bool instantiateInWorldSpace, bool bRegisterUndo)
		{
			GameObject newGO = null;
#if UNITY_5_4_OR_NEWER
			newGO = GameObject.Instantiate(sourceGameObject, parentTransform, instantiateInWorldSpace);
#else
			newGO = GameObject.Instantiate(sourceGameObject);
			newGO.transform.parent = parentTransform;
#endif

#if UNITY_EDITOR
			if (bRegisterUndo)
			{
				Undo.RegisterCreatedObjectUndo(newGO, "Instantiated " + newGO.name);
			}
#endif
			return newGO;
		}

		public static Component AddComponent<T>(GameObject target, bool bRegisterUndo)
		{
#if UNITY_EDITOR
			if (bRegisterUndo)
			{
				return Undo.AddComponent(target, typeof(T));
			}
			else
#endif
			{
				return target.AddComponent(typeof(T));
			}
		}

		public static void UndoRecordObject(Object objectToUndo, string name)
		{
#if UNITY_EDITOR
			Undo.RecordObject(objectToUndo, name);
#endif
		}

		public static void UndoCollapseCurrentGroup()
		{
#if UNITY_EDITOR
			Undo.CollapseUndoOperations(Undo.GetCurrentGroup());
#endif
		}

		/// <summary>
		/// Calculates and returns a list of all assets that obj depends on.
		/// </summary>
		/// <param name="obj"></param>
		/// <returns></returns>
		public static Object[] CollectDependencies(Object obj)
		{
#if UNITY_EDITOR
			return EditorUtility.CollectDependencies(new Object[] { obj });
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		/// <summary>
		/// Returns true if obj is stored on disk
		/// </summary>
		/// <param name="obj"></param>
		/// <returns></returns>
		public static bool IsPersistant(UnityEngine.Object obj)
		{
#if UNITY_EDITOR
			return EditorUtility.IsPersistent(obj);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return false;
#endif
		}

		/// <summary>
		/// Returns unique name based on siblings. If given name is already found
		/// adds integer to end and increments until found unique.
		/// </summary>
		/// <param name="parentTransform">Target parent for a new GameObject. Null means root level</param>
		/// <param name="name">Requested name for a new GameObject</param>
		/// <returns>Unique name for sibling gameobject</returns>
		public static string GetUniqueNameForSibling(Transform parentTransform, string name)
		{
#if UNITY_EDITOR
			return GameObjectUtility.GetUniqueNameForSibling(parentTransform, name);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
			return null;
#endif
		}

		/// <summary>
		/// Displays or updates a progress bar.
		/// </summary>
		/// <param name="title">Title of display</param>
		/// <param name="info">Info on display</param>
		/// <param name="progress">Progress ratio from 0 to 1</param>
		/// <returns></returns>
		public static void DisplayProgressBar(string title, string info, float progress)
		{
#if UNITY_EDITOR
			EditorUtility.DisplayProgressBar(title, info, progress);
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
#endif
		}

		/// <summary>
		/// Removes the progress bar on display
		/// </summary>
		public static void ClearProgressBar()
		{
#if UNITY_EDITOR
			EditorUtility.ClearProgressBar();
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
#endif
		}

		/// <summary>
		/// Returns true if we are in Editor, and we are not in play mode nor going into play mode.
		/// </summary>
		public static bool IsEditorNotInPlayModeAndNotGoingToPlayMode()
		{
#if UNITY_EDITOR
			return Application.isEditor && !UnityEditor.EditorApplication.isPlayingOrWillChangePlaymode && (Time.timeSinceLevelLoad > 0 || !Application.isPlaying);
#else
			return Application.isEditor && !Application.isPlaying;
#endif
		}

		/// <summary>
		/// Display message boxes in the editor.
		/// </summary>
		/// <param name="title"></param>
		/// <param name="message"></param>
		/// <param name="ok"></param>
		/// <param name="cancel"></param>
		/// <returns>True if OK button is pressed.</returns>
		public static bool DisplayDialog(string title, string message, string ok, string cancel = "")
		{
#if UNITY_EDITOR
			return EditorUtility.DisplayDialog(title, message, ok, cancel);
#else
			Debug.Log(string.Format("{0}: {1}", title, message));
			return true;
#endif
		}

		/// <summary>
		/// Display error message boxes in the editor.
		/// </summary>
		/// <param name="title"></param>
		/// <param name="message"></param>
		/// <param name="ok"></param>
		/// <param name="cancel"></param>
		/// <returns>True if OK button is pressed.</returns>
		public static bool DisplayErrorDialog(string title, string message, string ok, string cancel = "")
		{
#if UNITY_EDITOR
			return EditorUtility.DisplayDialog(string.Format("{0}: {1}", HEU_Defines.HEU_ERROR_TITLE, title), message, ok, cancel);
#else
			Debug.Log(string.Format("{0}: {1} - {2}", HEU_Constants.HEU_ERROR_TITLE, title, message));
			return true;
#endif
		}

#if UNITY_EDITOR
		public static SerializedProperty GetSerializedProperty(SerializedObject serializedObject, string propertyName)
		{
			SerializedProperty foundProperty = serializedObject.FindProperty(propertyName);
			if(foundProperty == null)
			{
				string errorMsg = string.Format("Property {0} not found on Object {1}. Make sure class {1} has member variable {0}.", propertyName, serializedObject.targetObject.GetType().ToString());
				Debug.LogAssertion(errorMsg);
			}
			return foundProperty;
		}
#endif

#if UNITY_EDITOR
		public static bool EditorDrawSerializedProperty(SerializedObject serializedObject, string propertyName, string label = null, bool bIncludeChildren = true)
		{
			SerializedProperty serializedProperty = GetSerializedProperty(serializedObject, propertyName);
			if(serializedProperty != null)
			{
				if(label == null)
				{
					EditorGUILayout.PropertyField(serializedProperty, bIncludeChildren);
				}
				else if(label.Length == 0)
				{
					EditorGUILayout.PropertyField(serializedProperty, GUIContent.none, bIncludeChildren);
				}
				else
				{
					EditorGUILayout.PropertyField(serializedProperty, new GUIContent(label), bIncludeChildren);
				}
				return true;
			}
			return false;
		}

		public static bool EditorDrawFloatProperty(SerializedObject serializedObject, string propertyName, string label = null)
		{
			SerializedProperty serializedProperty = GetSerializedProperty(serializedObject, propertyName);
			if (serializedProperty != null)
			{
				if (label == null)
				{
					EditorGUILayout.DelayedFloatField(serializedProperty);
				}
				else if (label.Length == 0)
				{
					EditorGUILayout.DelayedFloatField(serializedProperty, GUIContent.none);
				}
				else
				{
					EditorGUILayout.DelayedFloatField(serializedProperty, new GUIContent(label));
				}
				return true;
			}
			return false;
		}

		public static bool EditorDrawIntProperty(SerializedObject serializedObject, string propertyName, string label = null)
		{
			SerializedProperty serializedProperty = GetSerializedProperty(serializedObject, propertyName);
			if (serializedProperty != null)
			{
				if (label == null)
				{
					EditorGUILayout.DelayedIntField(serializedProperty);
				}
				else if (label.Length == 0)
				{
					EditorGUILayout.DelayedIntField(serializedProperty, GUIContent.none);
				}
				else
				{
					EditorGUILayout.DelayedIntField(serializedProperty, new GUIContent(label));
				}
				return true;
			}
			return false;
		}

		public static bool EditorDrawTextProperty(SerializedObject serializedObject, string propertyName, string label = null)
		{
			SerializedProperty serializedProperty = GetSerializedProperty(serializedObject, propertyName);
			if (serializedProperty != null)
			{
				if (label == null)
				{
					EditorGUILayout.DelayedTextField(serializedProperty);
				}
				else if (label.Length == 0)
				{
					EditorGUILayout.DelayedTextField(serializedProperty, GUIContent.none);
				}
				else
				{
					EditorGUILayout.DelayedTextField(serializedProperty, new GUIContent(label));
				}
				return true;
			}
			return false;
		}

		public static void EditorDrawIntArray(ref int[] intValues, string label = null)
		{
			// Arrays are drawn with a label, and rows of values.

			GUILayout.BeginHorizontal();
			{
				if(!string.IsNullOrEmpty(label))
				{
					EditorGUILayout.PrefixLabel(label);
				}

				GUILayout.BeginVertical(EditorStyles.helpBox);
				{
					int numElements = intValues.Length;
					int maxElementsPerRow = 4;

					GUILayout.BeginHorizontal();
					{
						for (int i = 0; i < numElements; ++i)
						{
							if (i > 0 && i % maxElementsPerRow == 0)
							{
								GUILayout.EndHorizontal();
								GUILayout.BeginHorizontal();
							}

							intValues[i] = EditorGUILayout.DelayedIntField(intValues[i]);
						}
					}
					GUILayout.EndHorizontal();
				}
				GUILayout.EndVertical();
			}
			GUILayout.EndHorizontal();
		}

		public static void EditorDrawFloatArray(ref float[] floatFields, string label = null)
		{
			// Arrays are drawn with a label, and rows of values.

			GUILayout.BeginHorizontal();
			{
				if (!string.IsNullOrEmpty(label))
				{
					EditorGUILayout.PrefixLabel(label);
				}

				GUILayout.BeginVertical(EditorStyles.helpBox);
				{
					int numElements = floatFields.Length;
					int maxElementsPerRow = 4;

					GUILayout.BeginHorizontal();
					{
						for (int i = 0; i < numElements; ++i)
						{
							if (i > 0 && i % maxElementsPerRow == 0)
							{
								GUILayout.EndHorizontal();
								GUILayout.BeginHorizontal();
							}

							floatFields[i] = EditorGUILayout.DelayedFloatField(floatFields[i]);
						}
					}
					GUILayout.EndHorizontal();
				}
				GUILayout.EndVertical();
			}
			GUILayout.EndHorizontal();
		}

		public static void EditorDrawTextArray(ref string[] stringFields, string label = null)
		{
			// Arrays are drawn with a label, and rows of values.

			GUILayout.BeginHorizontal();
			{
				if (!string.IsNullOrEmpty(label))
				{
					EditorGUILayout.PrefixLabel(label);
				}

				GUILayout.BeginVertical(EditorStyles.helpBox);
				{
					int numElements = stringFields.Length;
					int maxElementsPerRow = 4;

					GUILayout.BeginHorizontal();
					{
						for (int i = 0; i < numElements; ++i)
						{
							if (i > 0 && i % maxElementsPerRow == 0)
							{
								GUILayout.EndHorizontal();
								GUILayout.BeginHorizontal();
							}

							stringFields[i] = EditorGUILayout.DelayedTextField(stringFields[i]);
						}
					}
					GUILayout.EndHorizontal();
				}
				GUILayout.EndVertical();
			}
			GUILayout.EndHorizontal();
		}

		public static void EditorDrawArrayProperty(SerializedProperty arrayProperty, EditorDrawPropertyDelegate drawDelegate, string label = null)
		{
			// Arrays are drawn with a label, and rows of values.

			GUILayout.BeginHorizontal();
			{
				if (!string.IsNullOrEmpty(label))
				{
					EditorGUILayout.PrefixLabel(label);
				}

				GUILayout.BeginVertical(EditorStyles.helpBox);
				{
					int numElements = arrayProperty.arraySize;
					int maxElementsPerRow = 4;

					GUILayout.BeginHorizontal();
					{
						for (int i = 0; i < numElements; ++i)
						{
							if (i > 0 && i % maxElementsPerRow == 0)
							{
								GUILayout.EndHorizontal();
								GUILayout.BeginHorizontal();
							}

							SerializedProperty elementProperty = arrayProperty.GetArrayElementAtIndex(i);
							drawDelegate(elementProperty);
						}
					}
					GUILayout.EndHorizontal();
				}
				GUILayout.EndVertical();
			}
			GUILayout.EndHorizontal();
		}

		public delegate void EditorDrawPropertyDelegate(SerializedProperty property);

		public static void EditorDrawIntProperty(SerializedProperty property)
		{
			property.intValue = EditorGUILayout.DelayedIntField(property.intValue);
		}

		public static void EditorDrawFloatProperty(SerializedProperty property)
		{
			property.floatValue = EditorGUILayout.DelayedFloatField(property.floatValue);
		}

		public static void EditorDrawTextProperty(SerializedProperty property)
		{
			property.stringValue = EditorGUILayout.DelayedTextField(property.stringValue);
		}

		public static int[] GetSerializedPropertyArrayValuesInt(SerializedProperty property)
		{
			int[] array = null;
			if (property.isArray)
			{
				array = new int[property.arraySize];
				for (int i = 0; i < array.Length; ++i)
				{
					array[i] = property.GetArrayElementAtIndex(i).intValue;
				}
			}
			return array;
		}

		public static float[] GetSerializedPropertyArrayValuesFloat(SerializedProperty property)
		{
			float[] array = null;
			if (property.isArray)
			{
				array = new float[property.arraySize];
				for (int i = 0; i < array.Length; ++i)
				{
					array[i] = property.GetArrayElementAtIndex(i).floatValue;
				}
			}
			return array;
		}

		public static string[] GetSerializedPropertyArrayValuesString(SerializedProperty property)
		{
			string[] array = null;
			if (property.isArray)
			{
				array = new string[property.arraySize];
				for (int i = 0; i < array.Length; ++i)
				{
					array[i] = property.GetArrayElementAtIndex(i).stringValue;
				}
			}
			return array;
		}
#endif

		/// <summary>
		/// Sets given object to require update in Unity.
		/// Used for forcing Update/LateUpdate to be called on objects in Editor.
		/// Only works in Editor currently.
		/// </summary>
		/// <param name="obj">Object to set for update</param>
		public static void SetObjectDirtyForEditorUpdate(Object obj)
		{
#if UNITY_EDITOR
			EditorUtility.SetDirty(obj);
#endif
		}
	}

}   // HoudiniEngineUnity
