﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using System.Runtime.InteropServices;

#if UNITY_EDITOR
using UnityEditor;
#endif


namespace HoudiniEngineUnity
{
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Typedefs (copy these from HEU_Common.cs)
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;
	using HAPI_StringHandle = System.Int32;



	/// <summary>
	/// General utility functions for the Houdini Engine plugin.
	/// </summary>
	public class HEU_GeneralUtility
	{

		// ARRAYS GET -------------------------------------------------------------------------------------------------

		public delegate bool GetArray1ArgDel<T>(int arg1, [Out] T[] data, int start, int length);
		public delegate bool GetArray2ArgDel<ARG2, T>(int arg1, ARG2 arg2, [Out] T[] data, int start, int length);

		public static bool GetArray1Arg<T>(int arg1, GetArray1ArgDel<T> func, [Out] T[] data, int start, int count)
		{
			return GetArray(arg1, 0, func, null, data, start, count, 1);
		}

		public static bool GetArray2Arg<ARG2, T>(int arg1, ARG2 arg2, GetArray2ArgDel<ARG2, T> func, [Out] T[] data, int start, int count)
		{
			return GetArray(arg1, arg2, null, func, data, start, count, 1);
		}

		private static bool GetArray<ARG2, T>(
			int arg1, ARG2 arg2,
			GetArray1ArgDel<T> func1,
			GetArray2ArgDel<ARG2, T> func2,
			[Out] T[] data, int start, int count, int tupleSize)
		{
			int maxArraySize = HEU_Defines.HAPI_MAX_PAGE_SIZE / (Marshal.SizeOf(typeof(T)) * tupleSize);
			int localCount = count;
			int currentIndex = start;

			bool bResult = true;
			while (localCount > 0)
			{
				int length = 0;
				if (localCount > maxArraySize)
				{
					length = maxArraySize;
					localCount -= maxArraySize;
				}
				else
				{
					length = localCount;
					localCount = 0;
				}

				T[] localArray = new T[length * tupleSize];

				if (func1 != null)
				{
					bResult = func1(arg1, localArray, currentIndex, length);
				}
				else if (func2 != null)
				{
					bResult = func2(arg1, arg2, localArray, currentIndex, length);
				}
				else
				{
					HEU_HAPIUtility.LogError("No valid delegates given to GetArray< T >!");
					return false;
				}

				if (!bResult)
				{
					break;
				}

				// Copy from temporary array
				for (int i = currentIndex; i < (currentIndex + length); ++i)
				{
					for (int j = 0; j < tupleSize; ++j)
					{
						data[i * tupleSize + j] = localArray[(i - currentIndex) * tupleSize + j];
					}
				}

				currentIndex += length;
			}

			return bResult;
		}

		// ARRAYS SET -------------------------------------------------------------------------------------------------

		public static bool SetArray1Arg<T>(int arg1, GetArray1ArgDel<T> func, [Out] T[] data, int start, int count)
		{
			return SetArray(arg1, 0, func, null, data, start, count, 1);
		}

		public static bool SetArray2Arg<ARG2, T>(int arg1, ARG2 arg2, GetArray2ArgDel<ARG2, T> func, [Out] T[] data, int start, int count)
		{
			return SetArray(arg1, arg2, null, func, data, start, count, 1);
		}

		public static bool SetArray<ARG2, T>(
			int arg1, ARG2 arg2,
			GetArray1ArgDel<T> func1,
			GetArray2ArgDel<ARG2, T> func2,
			[Out] T[] data, int start, int count, int tupleSize)
		{
			int maxArraySize = HEU_Defines.HAPI_MAX_PAGE_SIZE / (Marshal.SizeOf(typeof(T)) * tupleSize);

			int localCount = count;
			int currentIndex = start;

			bool bResult = true;
			while (localCount > 0)
			{
				int length = 0;
				if (localCount > maxArraySize)
				{
					length = maxArraySize;
					localCount -= maxArraySize;
				}
				else
				{
					length = localCount;
					localCount = 0;
				}

				T[] localArray = new T[length * tupleSize];

				// Copy from main array to temporary
				for (int i = currentIndex; i < (currentIndex + length); ++i)
				{
					for (int j = 0; j < tupleSize; ++j)
					{
						localArray[(i - currentIndex) * tupleSize + j] = data[i * tupleSize + j];
					}
				}

				if (func1 != null)
				{
					bResult = func1(arg1, localArray, currentIndex, length);
				}
				else if (func2 != null)
				{
					bResult = func2(arg1, arg2, localArray, currentIndex, length);
				}
				else
				{
					HEU_HAPIUtility.LogError("No valid delegates given to SetArray<T>!");
					return false;
				}

				if (!bResult)
				{
					break;
				}

				currentIndex += length;
			}

			return bResult;
		}


		// ARRAYS GENERAL ---------------------------------------------------------------------------------------------

		/// <summary>
		/// Returns true if the element values in the two arrays match, and lengths match, and neither is null.
		/// </summary>
		/// <typeparam name="T">Type of array elements</typeparam>
		/// <param name="array1">First array</param>
		/// <param name="array2">Second array</param>
		/// <returns>True if array elements match</returns>
		public static bool DoArrayElementsMatch<T>(T[] array1, T[] array2)
		{
			if(ReferenceEquals(array1, array2))
			{
				return true;
			}

			if(array1 == null || array2 == null)
			{
				return false;
			}

			if(array1.Length != array2.Length)
			{
				return false;
			}

			EqualityComparer<T> equalityComparer = EqualityComparer<T>.Default;
			for(int i = 0; i < array1.Length; ++i)
			{
				if(!equalityComparer.Equals(array1[i], array2[i]))
				{
					return false;
				}
			}
			return true;
		}

		/// <summary>
		/// Returns true if the element values in the two arrays match, and lengths match, and neither is null.
		/// </summary>
		/// <typeparam name="T">Type of array elements</typeparam>
		/// <param name="array1">First array</param>
		/// <param name="startOffset1">Offset into first array to start checking</param>
		/// <param name="array2">Second array</param>
		/// <param name="startOffset2">Offset into second array to start checking</param>
		/// <param name="length">Number of elements to check</param>
		/// <returns>True if array elements match</returns>
		public static bool DoArrayElementsMatch<T>(T[] array1, int startOffset1, T[] array2, int startOffset2, int length)
		{
			if (array1 == null || array2 == null)
			{
				return false;
			}

			int lastIndex1 = startOffset1 + length;
			int lastIndex2 = startOffset2 + length;

			EqualityComparer<T> equalityComparer = EqualityComparer<T>.Default;
			for (int index1 = startOffset1, index2 = startOffset2; (index1 < lastIndex1) && (index2 < lastIndex2); ++index1, ++index2)
			{
				if (!equalityComparer.Equals(array1[index1], array2[index2]))
				{
					return false;
				}
			}
			return true;
		}


		// ATTRIBUTES -------------------------------------------------------------------------------------------------

		public delegate bool GetAttributeArrayInputFunc<T>(HAPI_NodeId geoID, HAPI_PartId partID, string name, ref HAPI_AttributeInfo info, [Out] T[] items, int start, int end);

		public static bool GetAttributeArray<T>(HAPI_NodeId geoID, HAPI_PartId partID, string name, ref HAPI_AttributeInfo info, T[] items, GetAttributeArrayInputFunc<T> getFunc, int count)
		{
			int maxArraySize = HEU_Defines.HAPI_MAX_PAGE_SIZE / (Marshal.SizeOf(typeof(T)) * info.tupleSize);

			int localCount = count;
			int currentIndex = 0;

			bool bResult = false;

			while (localCount > 0)
			{
				int length = 0;
				if (localCount > maxArraySize)
				{
					length = maxArraySize;
					localCount -= maxArraySize;
				}
				else
				{
					length = localCount;
					localCount = 0;
				}

				T[] localArray = new T[length * info.tupleSize];
				bResult = getFunc(geoID, partID, name, ref info, localArray, currentIndex, length);
				if (!bResult)
				{
					break;
				}

				// Copy data from temporary array
				for (int i = currentIndex; i < currentIndex + length; ++i)
				{
					for (int j = 0; j < info.tupleSize; ++j)
					{
						items[i * info.tupleSize + j] = localArray[(i - currentIndex) * info.tupleSize + j];
					}
				}

				currentIndex += length;
			}

			return bResult;
		}

		public static void GetAttribute<T>(HEU_SessionBase session, HAPI_NodeId geoID, HAPI_PartId partID, string name, ref HAPI_AttributeInfo info, ref T[] data, GetAttributeArrayInputFunc<T> getFunc)
		{
			int originalTupleSize = info.tupleSize;
			bool bResult = false;
			for (HAPI_AttributeOwner type = 0; type < HAPI_AttributeOwner.HAPI_ATTROWNER_MAX; ++type)
			{
				bResult = session.GetAttributeInfo(geoID, partID, name, type, ref info);
				if (bResult && info.exists)
				{
					break;
				}
			}

			if (!bResult || !info.exists)
			{
				return;
			}

			if (originalTupleSize > 0)
			{
				info.tupleSize = originalTupleSize;
			}

			data = new T[info.count * info.tupleSize];
			GetAttributeArray(geoID, partID, name, ref info, data, getFunc, info.count);
		}

		public static void GetAttributeStringDataHelper(HEU_SessionBase session, HAPI_NodeId geoID, HAPI_PartId partID, string name, ref HAPI_AttributeInfo info, ref HAPI_StringHandle[] data)
		{
			int originalTupleSize = info.tupleSize;
			bool bResult = false;
			for (HAPI_AttributeOwner type = 0; type < HAPI_AttributeOwner.HAPI_ATTROWNER_MAX; ++type)
			{
				bResult = session.GetAttributeInfo(geoID, partID, name, type, ref info);
				if (bResult && info.exists)
				{
					break;
				}
			}

			if (!bResult || !info.exists)
			{
				return;
			}

			if (originalTupleSize > 0)
			{
				info.tupleSize = originalTupleSize;
			}

			data = new HAPI_StringHandle[info.count * info.tupleSize];
			bResult = session.GetAttributeStringData(geoID, partID, name, ref info, data, 0, info.count);
			if(!bResult)
			{
				Debug.LogErrorFormat("Failed to get string IDs for attribute {0}", name);
			}
		}

		public delegate bool SetAttributeArrayFunc<T>(HAPI_NodeId geoID, HAPI_PartId partID, string attrName, ref HAPI_AttributeInfo attrInfo, T[] items, int start, int end);

		public static bool SetAttributeArray<T>(HAPI_NodeId geoID, HAPI_PartId partID, string attrName, ref HAPI_AttributeInfo attrInfo, T[] items, SetAttributeArrayFunc<T> setFunc, int count)
		{
			bool bResult = false;

			int maxArraySize = 0;
			if(typeof(T) == typeof(string))
			{
				int maxStringLength = 1;
				foreach(T s in items)
				{
					string str = (string)(object)s;
					if(str.Length > maxStringLength)
					{
						maxStringLength = str.Length;
					}
				}
				maxArraySize = HEU_Defines.HAPI_MAX_PAGE_SIZE / (maxStringLength * Marshal.SizeOf(typeof(char)) * attrInfo.tupleSize);
			}
			else
			{
				maxArraySize = HEU_Defines.HAPI_MAX_PAGE_SIZE / (Marshal.SizeOf(typeof(T)) * attrInfo.tupleSize);
			}

			int localCount = count;
			int currentIndex = 0;

			while(localCount > 0)
			{
				int length = 0;
				if(localCount > maxArraySize)
				{
					length = maxArraySize;
					localCount -= maxArraySize;
				}
				else
				{
					length = localCount;
					localCount = 0;
				}

				T[] localArray = new T[length * attrInfo.tupleSize];

				// Copy subset to temp array
				for(int i = currentIndex; i < currentIndex + length; ++i)
				{
					for(int j = 0; j < attrInfo.tupleSize; ++j)
					{
						localArray[(i - currentIndex) * attrInfo.tupleSize + j] = items[i * attrInfo.tupleSize + j];
					}
				}

				bResult = setFunc(geoID, partID, attrName, ref attrInfo, localArray, currentIndex, length);
				if (!bResult)
				{
					break;
				}

				currentIndex += length;
			}

			return bResult;
		}

		public static bool SetAttribute<T>(HAPI_NodeId geoID, HAPI_PartId partID, string attrName, ref HAPI_AttributeInfo attrInfo, T[] items, SetAttributeArrayFunc<T> setFunc)
		{
			return SetAttributeArray(geoID, partID, attrName, ref attrInfo, items, setFunc, attrInfo.count);
		}

		public static bool CheckAttributeExists(HEU_SessionBase session, HAPI_NodeId geoID, HAPI_PartId partID, string attribName, HAPI_AttributeOwner attribOwner)
		{
			HAPI_AttributeInfo attribInfo = new HAPI_AttributeInfo();
			if(session.GetAttributeInfo(geoID, partID, attribName, attribOwner, ref attribInfo))
			{
				return attribInfo.exists;
			}
			return false;
		}

		public static bool GetAttributeInfo(HEU_SessionBase session, HAPI_NodeId geoID, HAPI_PartId partID, string attribName, ref HAPI_AttributeInfo attribInfo)
		{
			bool bResult = false;
			for (HAPI_AttributeOwner type = 0; type < HAPI_AttributeOwner.HAPI_ATTROWNER_MAX; ++type)
			{
				bResult = session.GetAttributeInfo(geoID, partID, attribName, type, ref attribInfo);
				if (!bResult)
				{
					attribInfo.exists = false;
					return false;
				}
				else if(attribInfo.exists)
				{
					break;
				}
			}
			return true;
		}

		/// <summary>
		/// Copy the world transform values from src to dest.
		/// </summary>
		/// <param name="src"></param>
		/// <param name="dest"></param>
		public static void CopyWorldTransformValues(Transform src, Transform dest)
		{
			dest.localScale = src.localScale;
			dest.rotation = src.rotation;
			dest.position = src.position;
		}

		/// <summary>
		/// Copy the local transform values from src to dest.
		/// </summary>
		/// <param name="src"></param>
		/// <param name="dest"></param>
		public static void CopyLocalTransformValues(Transform src, Transform dest)
		{
			dest.localScale = src.localScale;
			dest.localRotation = src.localRotation;
			dest.localPosition = src.localPosition;
		}

		/// <summary>
		/// Returns list of child gameobjects of parentGO.
		/// </summary>
		/// <param name="parentGO"></param>
		/// <returns>List of child gameobjects of parentGO</returns>
		public static List<GameObject> GetChildGameObjects(GameObject parentGO)
		{
			Transform parentTransform = parentGO.transform;
			List<GameObject> children = new List<GameObject>();

			foreach (Transform child in parentTransform)
			{
				children.Add(child.gameObject);
			}
			return children;
		}

		/// <summary>
		/// Returns list of child gameobjects of parentGO with name containing the given patter.
		/// Or if bExcluse is true, then the inverse of above list.
		/// </summary>
		/// <param name="parentGO">The parent gameobject to get children from.</param>
		/// <param name="pattern">The pattern to search for in the game.</param>
		/// <param name="bExclude">If true, returns list of children with names not containing the pattern.</param>
		/// <returns></returns>
		public static List<GameObject> GetChildGameObjectsWithNamePattern(GameObject parentGO, string pattern, bool bExclude)
		{
			Transform parentTransform = parentGO.transform;
			List<GameObject> children = new List<GameObject>();

			foreach (Transform child in parentTransform)
			{
				string childName = child.name;

				if (System.Text.RegularExpressions.Regex.IsMatch(childName, pattern) != bExclude)
				{
					children.Add(child.gameObject);
				}
			}
			return children;
		}

		/// <summary>
		/// Returns list of child gameobjects of parentGO that are instances.
		/// </summary>
		/// <param name="parentGO">The parent gameobject to get children from.</param>
		/// <returns></returns>
		public static List<GameObject> GetInstanceChildObjects(GameObject parentGO)
		{
			return GetChildGameObjectsWithNamePattern(parentGO, HEU_Defines.HEU_INSTANCE_NAME_PATTERN, false);
		}

		/// <summary>
		/// Returns list of child gameobjects of parentGO that are not instances.
		/// </summary>
		/// <param name="parentGO">The parent gameobject to get children from.</param>
		/// <returns></returns>
		public static List<GameObject> GetNonInstanceChildObjects(GameObject parentGO)
		{
			return GetChildGameObjectsWithNamePattern(parentGO, HEU_Defines.HEU_INSTANCE_NAME_PATTERN, true);
		}

		/// <summary>
		/// Returns the gameobject with the name from given list.
		/// </summary>
		/// <param name="goList">List to search</param>
		/// <param name="name">Name to match</param>
		/// <returns>Found gameobject or null if not found</returns>
		public static GameObject GetGameObjectByName(List<GameObject> goList, string name)
		{
			foreach (GameObject go in goList)
			{
				if(go.name.Equals(name))
				{
					return go;
				}
			}
			return null;
		}

		/// <summary>
		/// Gets existing or creates new component for given gameObject.
		/// </summary>
		/// <typeparam name="T">Component to retrieve (and/or create)</typeparam>
		/// <param name="gameObject">GameObject to retrieve (and/or create) from</param>
		/// <returns>Component part of the gameObject</returns>
		public static T GetOrCreateComponent<T>(GameObject gameObject) where T : Component
		{
			T component = gameObject.GetComponent<T>();
			if (component == null)
			{
				component = gameObject.AddComponent<T>();
			}
			return component;
		}

		/// <summary>
		/// Destroy any potential generated components on given gameObject
		/// </summary>
		/// <param name="gameObject"></param>
		public static void DestroyGeneratedComponents(GameObject gameObject)
		{
			DestroyComponent<MeshFilter>(gameObject);
			DestroyComponent<MeshRenderer>(gameObject);
			DestroyComponent<Collider>(gameObject);
			DestroyComponent<TerrainCollider>(gameObject);
			DestroyComponent<Terrain>(gameObject);
		}

		/// <summary>
		/// Destroys component T if found on gameObject.
		/// </summary>
		/// <typeparam name="T">Component type</typeparam>
		/// <param name="gameObject">GameObject to search component on</param>
		public static void DestroyComponent<T>(GameObject gameObject) where T : Component
		{
			T component = gameObject.GetComponent<T>();
			if (component != null)
			{
				DestroyImmediate(component);
			}
		}

		/// <summary>
		/// Destroys obj immediately and permanently.
		/// </summary>
		/// <param name="obj">The object to destroy</param>
		/// <param name="bAllowDestroyingAssets">Force destroy asset</param>
		public static void DestroyImmediate(UnityEngine.Object obj, bool bAllowDestroyingAssets = false, bool bRegisterUndo = false)
		{
#if UNITY_EDITOR
			if (bRegisterUndo)
			{
				Undo.DestroyObjectImmediate(obj);
			}
			else
			{
				UnityEngine.Object.DestroyImmediate(obj, bAllowDestroyingAssets);
			}
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
#endif
		}

		public static void DestroyBakedGameObjects(List<GameObject> gameObjectsToDestroy)
		{
			DestroyBakedGameObjectsWithEndName(gameObjectsToDestroy, null);
		}

		public static void DestroyBakedGameObjectsWithEndName(List<GameObject> gameObjectsToDestroy, string endName)
		{
			int numLeft = gameObjectsToDestroy.Count;
			for (int i = 0; i < numLeft; ++i)
			{
				GameObject deleteGO = gameObjectsToDestroy[i];
				if (string.IsNullOrEmpty(endName) || deleteGO.name.EndsWith(endName))
				{
					gameObjectsToDestroy[i] = null;
					HEU_PartData.DestroyExistingGeneratedComponentsMeshData(deleteGO, true);
					HEU_GeneralUtility.DestroyImmediate(deleteGO);
				}
			}
		}

		/// <summary>
		/// Set the given gameobject's render visibility
		/// </summary>
		/// <param name="gameObject">Gameobject to set visibility on</param>
		/// <param name="bVisible">Visibility state</param>
		public static void SetGameObjectRenderVisiblity(GameObject gameObject, bool bVisible)
		{
			MeshRenderer partMeshRenderer = gameObject.GetComponent<MeshRenderer>();
			if (partMeshRenderer != null)
			{
				partMeshRenderer.enabled = bVisible;
			}
		}

		public static string ColorToString(Color c)
		{
			return string.Format("{0},{1},{2},{3}", c[0], c[1], c[2], c[3]);
		}

		public static Color StringToColor(string colorString)
		{
			Color c = new Color();
			string[] strList = colorString.Split(',');
			for(int i = 0; i < strList.Length; ++i)
			{
				float f = 1f;
				if(float.TryParse(strList[i], out f))
				{
					c[i] = f;
				}
			}
			return c;
		}

		public static bool DoesUnityTagExist(string tagName)
		{
			// TODO: Implement this
			return true;// string.IsNullOrEmpty(tagName);
		}

		public static void SetLayer(GameObject rootGO, int layer, bool bIncludeChildren)
		{
			rootGO.layer = layer;
			if(bIncludeChildren)
			{
				foreach(Transform trans in rootGO.transform.GetComponentsInChildren<Transform>(true))
				{
					trans.gameObject.layer = layer;
				}
			}
		}

		public static bool IsMouseWithinSceneView(Camera camera, Vector2 mousePosition)
		{
			return (mousePosition.x < camera.pixelWidth && mousePosition.y < camera.pixelHeight
						&& mousePosition.x > 0 && mousePosition.y > 0);
		}

		public static bool IsMouseOverRect(Camera camera, Vector2 mousePosition, ref Rect rect)
		{
			mousePosition.y = camera.pixelHeight - mousePosition.y;
			return rect.Contains(mousePosition);
		}

		/// <summary>
		/// Returns the type from name.
		/// </summary>
		/// <param name="typeName">String name of type</param>
		/// <returns>Valid type or null if not found in loaded assemblies.</returns>
		public static System.Type GetSystemTypeByName(string typeName)
		{
#if UNITY_EDITOR
			System.Reflection.Assembly[] assemblies = System.AppDomain.CurrentDomain.GetAssemblies();
			foreach (System.Reflection.Assembly assembly in assemblies)
			{
				System.Type[] types = assembly.GetTypes();
				foreach (System.Type type in types)
				{
					if(type.Name.Equals(typeName))
					{
						return type;
					}
				}
			}
#else
			Debug.LogWarning(HEU_Constants.HEU_USERMSG_NONEDITOR_NOT_SUPPORTED);
#endif
			return null;
		}
	}


	public static class ArrayExtensions
	{
		/// <summary>
		/// Set the given array with the given value for every element.
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="array"></param>
		/// <param name="defaultValue"></param>
		public static void Init<T>(this T[] array, T defaultValue)
		{
			if (array != null)
			{
				for (int i = 0; i < array.Length; i++)
				{
					array[i] = defaultValue;
				}
			}
		}

		/// <summary>
		/// Set the given list with the given value for every element.
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="array"></param>
		/// <param name="defaultValue"></param>
		public static void Init<T>(this List<T> array, T defaultValue)
		{
			if(array != null)
			{
				for (int i = 0; i < array.Count; i++)
				{
					array[i] = defaultValue;
				}
			}
		}
	}

	public class ReverseCompare : IComparer
	{
		public int Compare(object x, object y)
		{
			return (new CaseInsensitiveComparer().Compare(y, x));
		}
	}

}   // HoudiniEngineUnity