﻿/*
* Copyright (c) <2018> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

// Uncomment to profile
//#define HEU_PROFILER_ON

using UnityEngine;
using System;
using System.Collections.Generic;


namespace HoudiniEngineUnity
{
	using HAPI_NodeId = System.Int32;
	using HAPI_PartId = System.Int32;
	using HAPI_StringHandle = System.Int32;


	/// <summary>
	/// Stores geometry and material info for a part that is then used to generate Unity geometry.
	/// </summary>
	public class HEU_GenerateGeoCache
	{
		public HAPI_NodeId GeoID { get { return _geoInfo.nodeId; } }
		public HAPI_PartId PartID { get { return _partInfo.id; } }

		public HAPI_GeoInfo _geoInfo;
		public HAPI_PartInfo _partInfo;

		public string _partName;

		public int[] _vertexList;

		public HAPI_NodeId[] _houdiniMaterialIDs;

		public bool _singleFaceUnityMaterial;
		public bool _singleFaceHoudiniMaterial;

		public Dictionary<int, HEU_UnityMaterialInfo> _unityMaterialInfos;
		public HAPI_AttributeInfo _unityMaterialAttrInfo;
		public HAPI_StringHandle[] _unityMaterialAttrName;
		public Dictionary<HAPI_StringHandle, string> _unityMaterialAttrStringsMap = new Dictionary<HAPI_StringHandle, string>();

		public HAPI_AttributeInfo _substanceMaterialAttrNameInfo;
		public HAPI_StringHandle[] _substanceMaterialAttrName;
		public Dictionary<HAPI_StringHandle, string> _substanceMaterialAttrStringsMap = new Dictionary<HAPI_StringHandle, string>();

		public HAPI_AttributeInfo _substanceMaterialAttrIndexInfo;
		public int[] _substanceMaterialAttrIndex;

		public List<HEU_MaterialData> _inUseMaterials = new List<HEU_MaterialData>();

		public HAPI_AttributeInfo _posAttrInfo;
		public HAPI_AttributeInfo _uvAttrInfo;
		public HAPI_AttributeInfo _uv2AttrInfo;
		public HAPI_AttributeInfo _uv3AttrInfo;
		public HAPI_AttributeInfo _normalAttrInfo;
		public HAPI_AttributeInfo _colorAttrInfo;
		public HAPI_AttributeInfo _alphaAttrInfo;
		public HAPI_AttributeInfo _tangentAttrInfo;

		public float[] _posAttr;
		public float[] _uvAttr;
		public float[] _uv2Attr;
		public float[] _uv3Attr;
		public float[] _normalAttr;
		public float[] _colorAttr;
		public float[] _alphaAttr;
		public float[] _tangentAttr;

		public string[] _groups;
		public bool _hasGroupGeometry;

		public Dictionary<string, int[]> _groupSplitVertexIndices = new Dictionary<string, int[]>();
		public Dictionary<string, List<int>> _groupSplitFaceIndices = new Dictionary<string, List<int>>();

		public int[] _allCollisionVertexList;
		public int[] _allCollisionFaceIndices;

		public List<VertexEntry>[] _sharedNormalIndices;
		public float _cosineThreshold;



		public static HEU_GenerateGeoCache GetPopulatedGeoCache(HEU_SessionBase session, HAPI_NodeId geoID, HAPI_PartId partID)
		{
#if HEU_PROFILER_ON
			float generateGeoCacheStartTime = Time.realtimeSinceStartup;
#endif

			HEU_GenerateGeoCache geoCache = new HEU_GenerateGeoCache();

			Debug.Assert(geoID != HEU_Defines.HEU_INVALID_NODE_ID, "Invalid Geo ID! Unable to update materials!");
			Debug.Assert(partID != HEU_Defines.HEU_INVALID_NODE_ID, "Invalid Part ID! Unable to update materials!");

			geoCache._geoInfo = new HAPI_GeoInfo();
			if (!session.GetGeoInfo(geoID, ref geoCache._geoInfo))
			{
				return null;
			}

			geoCache._partInfo = new HAPI_PartInfo();
			if (!session.GetPartInfo(geoID, partID, ref geoCache._partInfo))
			{
				return null;
			}

			geoCache._partName = HEU_SessionManager.GetString(geoCache._partInfo.nameSH, session);

			// Unity max indices limitation!
#if UNITY_2017_3_OR_NEWER
			uint UNITY_MAX_VERTS = uint.MaxValue;
#else
			uint UNITY_MAX_VERTS = ushort.MaxValue;
#endif
			uint maxVertexCount = Convert.ToUInt32(geoCache._partInfo.vertexCount);
			if (maxVertexCount > UNITY_MAX_VERTS)
			{
				Debug.LogErrorFormat("Part {0} has vertex count of {1} which is above Unity maximum of {2}.\nReduce this in Houdini.",
					geoCache._partName, maxVertexCount, UNITY_MAX_VERTS);
				return null;
			}

			geoCache._vertexList = new int[geoCache._partInfo.vertexCount];
			if (!HEU_GeneralUtility.GetArray2Arg(geoID, partID, session.GetVertexList, geoCache._vertexList, 0, geoCache._partInfo.vertexCount))
			{
				return null;
			}

			geoCache._houdiniMaterialIDs = new HAPI_NodeId[geoCache._partInfo.faceCount];
			if (!session.GetMaterialNodeIDsOnFaces(geoID, partID, ref geoCache._singleFaceHoudiniMaterial, geoCache._houdiniMaterialIDs, geoCache._partInfo.faceCount))
			{
				return null;
			}

			geoCache.PopulateUnityMaterialData(session);

			if (!geoCache.PopulateGeometryData(session))
			{
				return null;
			}

#if HEU_PROFILER_ON
			Debug.LogFormat("GENERATE GEO CACHE TIME:: {0}", (Time.realtimeSinceStartup - generateGeoCacheStartTime));
#endif

			return geoCache;
		}

		public void PopulateUnityMaterialData(HEU_SessionBase session)
		{
			// First we look for Unity and Substance material attributes on faces.
			// We fill up the following dictionary with unique Unity + Substance material information
			_unityMaterialInfos = new Dictionary<int, HEU_UnityMaterialInfo>();

			_unityMaterialAttrInfo = new HAPI_AttributeInfo();
			_unityMaterialAttrName = new HAPI_StringHandle[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_PluginSettings.UnityMaterialAttribName, ref _unityMaterialAttrInfo, ref _unityMaterialAttrName, session.GetAttributeStringData);

			// Store a local copy of the actual string values since the indices get overwritten by the next call to session.GetAttributeStringData.
			// Using a dictionary to only query the unique strings, as doing all of them is very slow and unnecessary.
			_unityMaterialAttrStringsMap = new Dictionary<HAPI_StringHandle, string>();
			foreach (HAPI_StringHandle strHandle in _unityMaterialAttrName)
			{
				if (!_unityMaterialAttrStringsMap.ContainsKey(strHandle))
				{
					string materialName = HEU_SessionManager.GetString(strHandle, session);
					if (string.IsNullOrEmpty(materialName))
					{
						// Warn user of empty string, but add it anyway to our map so we don't keep trying to parse it
						Debug.LogWarningFormat("Found empty material attribute value for part {0}.", _partName);
					}
					_unityMaterialAttrStringsMap.Add(strHandle, materialName);
					//Debug.LogFormat("Added Unity material: " + materialName);
				}
			}

			_substanceMaterialAttrNameInfo = new HAPI_AttributeInfo();
			_substanceMaterialAttrName = new HAPI_StringHandle[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_PluginSettings.UnitySubMaterialAttribName, ref _substanceMaterialAttrNameInfo, ref _substanceMaterialAttrName, session.GetAttributeStringData);

			_substanceMaterialAttrStringsMap = new Dictionary<HAPI_StringHandle, string>();
			foreach (HAPI_StringHandle strHandle in _substanceMaterialAttrName)
			{
				if (!_substanceMaterialAttrStringsMap.ContainsKey(strHandle))
				{
					string substanceName = HEU_SessionManager.GetString(strHandle, session);
					if (string.IsNullOrEmpty(substanceName))
					{
						// Warn user of empty string, but add it anyway to our map so we don't keep trying to parse it
						Debug.LogWarningFormat("Found invalid substance material attribute value ({0}) for part {1}.",
							_partName, substanceName);
					}
					_substanceMaterialAttrStringsMap.Add(strHandle, substanceName);
					//Debug.LogFormat("Added Substance material: " + substanceName);
				}
			}

			_substanceMaterialAttrIndexInfo = new HAPI_AttributeInfo();
			_substanceMaterialAttrIndex = new int[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_PluginSettings.UnitySubMaterialIndexAttribName, ref _substanceMaterialAttrIndexInfo, ref _substanceMaterialAttrIndex, session.GetAttributeIntData);


			if (_unityMaterialAttrInfo.exists)
			{
				if (_unityMaterialAttrInfo.owner == HAPI_AttributeOwner.HAPI_ATTROWNER_DETAIL && _unityMaterialAttrName.Length > 0)
				{
					CreateMaterialInfoEntryFromAttributeIndex(this, 0);

					// Detail unity material attribute means we can treat it as single material
					_singleFaceUnityMaterial = true;
				}
				else
				{
					for(HAPI_StringHandle i = 0; i < _unityMaterialAttrName.Length; ++i)
					{
						CreateMaterialInfoEntryFromAttributeIndex(this, i);
					}
				}
			}
		}

		public static int GetMaterialKeyFromAttributeIndex(HEU_GenerateGeoCache geoCache, int attributeIndex, out string unityMaterialName, out string substanceName, out int substanceIndex)
		{
			unityMaterialName = null;
			substanceName = null;
			substanceIndex = -1;
			if (attributeIndex < geoCache._unityMaterialAttrName.Length && geoCache._unityMaterialAttrStringsMap.TryGetValue(geoCache._unityMaterialAttrName[attributeIndex], out unityMaterialName))
			{
				if (geoCache._substanceMaterialAttrNameInfo.exists && geoCache._substanceMaterialAttrName.Length > 0)
				{
					geoCache._substanceMaterialAttrStringsMap.TryGetValue(geoCache._substanceMaterialAttrName[attributeIndex], out substanceName);
				}

				if (geoCache._substanceMaterialAttrIndexInfo.exists && string.IsNullOrEmpty(substanceName) && geoCache._substanceMaterialAttrIndex[attributeIndex] >= 0)
				{
					substanceIndex = geoCache._substanceMaterialAttrIndex[attributeIndex];
				}

				return HEU_MaterialFactory.GetUnitySubstanceMaterialKey(unityMaterialName, substanceName, substanceIndex);
			}
			return HEU_Defines.HEU_INVALID_MATERIAL;
		}

		public static void CreateMaterialInfoEntryFromAttributeIndex(HEU_GenerateGeoCache geoCache, int materialAttributeIndex)
		{
			string unityMaterialName = null;
			string substanceName = null;
			int substanceIndex = -1;
			int materialKey = GetMaterialKeyFromAttributeIndex(geoCache, materialAttributeIndex, out unityMaterialName, out substanceName, out substanceIndex);
			if (!geoCache._unityMaterialInfos.ContainsKey(materialKey))
			{
				geoCache._unityMaterialInfos.Add(materialKey, new HEU_UnityMaterialInfo(unityMaterialName, substanceName, substanceIndex));
			}
		}

		public bool PopulateGeometryData(HEU_SessionBase session)
		{
			// Get vertex position
			HAPI_AttributeInfo posAttrInfo = new HAPI_AttributeInfo();
			_posAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_POSITION, ref posAttrInfo, ref _posAttr, session.GetAttributeFloatData);
			if (!posAttrInfo.exists)
			{
				return false;
			}
			else if (posAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_POINT)
			{
				Debug.LogErrorFormat("{0} only supports position as POINT attribute. Position attribute of {1} type not supported!", HEU_Defines.HEU_PRODUCT_NAME, posAttrInfo.owner);
				return false;
			}

			// Get UV attributes
			_uvAttrInfo = new HAPI_AttributeInfo();
			_uvAttrInfo.tupleSize = 2;
			_uvAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_UV, ref _uvAttrInfo, ref _uvAttr, session.GetAttributeFloatData);

			// Get UV2 attributes
			_uv2AttrInfo = new HAPI_AttributeInfo();
			_uv2AttrInfo.tupleSize = 2;
			_uv2Attr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_UV2, ref _uv2AttrInfo, ref _uv2Attr, session.GetAttributeFloatData);

			// Get UV3 attributes
			_uv3AttrInfo = new HAPI_AttributeInfo();
			_uv3AttrInfo.tupleSize = 2;
			_uv3Attr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_UV3, ref _uv3AttrInfo, ref _uv3Attr, session.GetAttributeFloatData);

			// Get normal attributes
			_normalAttrInfo = new HAPI_AttributeInfo();
			_normalAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_NORMAL, ref _normalAttrInfo, ref _normalAttr, session.GetAttributeFloatData);

			// Get colour attributes
			_colorAttrInfo = new HAPI_AttributeInfo();
			_colorAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_COLOR, ref _colorAttrInfo, ref _colorAttr, session.GetAttributeFloatData);

			// Get alpha attributes
			_alphaAttrInfo = new HAPI_AttributeInfo();
			_alphaAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_ALPHA, ref _alphaAttrInfo, ref _alphaAttr, session.GetAttributeFloatData);

			// Get tangent attributes
			_tangentAttrInfo = new HAPI_AttributeInfo();
			_tangentAttr = new float[0];
			HEU_GeneralUtility.GetAttribute(session, GeoID, PartID, HEU_Defines.HAPI_ATTRIB_TANGENT, ref _tangentAttrInfo, ref _tangentAttr, session.GetAttributeFloatData);

			// Warn user since we are splitting points by attributes, might prevent some attrributes
			// to be transferred over properly
			if (_normalAttrInfo.exists && _normalAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_POINT
									&& _normalAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_VERTEX)
			{
				Debug.LogWarningFormat("{0}: Normals are not declared as point or vertex attributes.\nSet them as per point or vertices in HDA.", _partName);
			}

			if (_tangentAttrInfo.exists && _tangentAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_POINT
									&& _tangentAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_VERTEX)
			{
				Debug.LogWarningFormat("{0}: Tangents are not declared as point or vertex attributes.\nSet them as per point or vertices in HDA.", _partName);
			}

			if (_colorAttrInfo.exists && _colorAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_POINT
									&& _colorAttrInfo.owner != HAPI_AttributeOwner.HAPI_ATTROWNER_VERTEX)
			{
				Debug.LogWarningFormat("{0}: Colours are not declared as point or vertex attributes."
					+ "\nCurrently set as owner type {1}. Set them as per point or vertices in HDA.", _partName, _colorAttrInfo.owner);
			}

			_groups = HEU_SessionManager.GetGroupNames(GeoID, HAPI_GroupType.HAPI_GROUPTYPE_PRIM);

			_allCollisionVertexList = new int[_vertexList.Length];
			_allCollisionFaceIndices = new int[_partInfo.faceCount];

			_hasGroupGeometry = false;

			// We go through each group, building up a triangle list of indices that belong to it
			// For strictly colliders (ie. non-rendering), we only create geometry colliders 
			for (int g = 0; g < _groups.Length; ++g)
			{
				string groupName = _groups[g];

				// Query HAPI to get the group membership. 
				// This is returned as an array of 1s for vertices that belong to this group.
				int[] membership = null;
				HEU_SessionManager.GetGroupMembership(session, GeoID, PartID, HAPI_GroupType.HAPI_GROUPTYPE_PRIM, groupName, ref membership);

				bool bIsCollidable = groupName.Contains(HEU_PluginSettings.CollisionGroupName);
				bool bIsRenderCollidable = groupName.Contains(HEU_PluginSettings.RenderedCollisionGroupName);

				if (bIsCollidable || bIsRenderCollidable)
				{
					// Extract vertex indices for this collision group

					int[] groupVertexList = new int[_vertexList.Length];
					groupVertexList.Init<int>(-1);

					int groupVertexListCount = 0;

					List<int> allFaceList = new List<int>();
					for (int f = 0; f < membership.Length; ++f)
					{
						if (membership[f] > 0)
						{
							// This face is a member of the specified group

							allFaceList.Add(f);

							groupVertexList[f * 3 + 0] = _vertexList[f * 3 + 0];
							groupVertexList[f * 3 + 1] = _vertexList[f * 3 + 1];
							groupVertexList[f * 3 + 2] = _vertexList[f * 3 + 2];

							// Mark vertices as used
							_allCollisionVertexList[f * 3 + 0] = 1;
							_allCollisionVertexList[f * 3 + 1] = 1;
							_allCollisionVertexList[f * 3 + 2] = 1;

							// Mark face as used
							_allCollisionFaceIndices[f] = 1;

							groupVertexListCount += 3;
						}
					}

					if (groupVertexListCount > 0)
					{
						_groupSplitVertexIndices.Add(groupName, groupVertexList);
						_groupSplitFaceIndices.Add(groupName, allFaceList);

						_hasGroupGeometry = true;

						//Debug.Log("Adding collision group: " + groupName + " with index count: " + _groupVertexList.Length);
					}
				}
			}

			if (_hasGroupGeometry)
			{
				// Construct vertex list for all other vertices that are not part of collision geometry
				int[] remainingGroupSplitFaces = new int[_vertexList.Length];
				remainingGroupSplitFaces.Init<int>(-1);
				bool bMainSplitGroup = false;

				List<int> remainingGroupSplitFaceIndices = new List<int>();

				for (int cv = 0; cv < _allCollisionVertexList.Length; ++cv)
				{
					if (_allCollisionVertexList[cv] == 0)
					{
						// Unused index, so add it to unused vertex list
						remainingGroupSplitFaces[cv] = _vertexList[cv];
						bMainSplitGroup = true;
					}
				}

				for (int cf = 0; cf < _allCollisionFaceIndices.Length; ++cf)
				{
					if (_allCollisionFaceIndices[cf] == 0)
					{
						remainingGroupSplitFaceIndices.Add(cf);
					}
				}

				if (bMainSplitGroup)
				{
					_groupSplitVertexIndices.Add(HEU_Defines.HEU_DEFAULT_GEO_GROUP_NAME, remainingGroupSplitFaces);
					_groupSplitFaceIndices.Add(HEU_Defines.HEU_DEFAULT_GEO_GROUP_NAME, remainingGroupSplitFaceIndices);

					//Debug.Log("Adding remaining group with index count: " + remainingGroupSplitFaces.Length);
				}
			}
			else
			{
				_groupSplitVertexIndices.Add(HEU_Defines.HEU_DEFAULT_GEO_GROUP_NAME, _vertexList);

				List<int> allFaces = new List<int>();
				for (int f = 0; f < _partInfo.faceCount; ++f)
				{
					allFaces.Add(f);
				}
				_groupSplitFaceIndices.Add(HEU_Defines.HEU_DEFAULT_GEO_GROUP_NAME, allFaces);

				//Debug.Log("Adding single non-group with index count: " + _vertexList.Length);
			}

			// We'll be generating the normals if they're not provided by Houdini
			// For every vertex, we'll hold list of other vertices that it shares faces with (ie. connected to)
			_sharedNormalIndices = null;
			_cosineThreshold = 0f;
			if (!_normalAttrInfo.exists)
			{
				_sharedNormalIndices = new List<VertexEntry>[_vertexList.Length];
				_cosineThreshold = Mathf.Cos(HEU_PluginSettings.NormalGenerationThresholdAngle * Mathf.Deg2Rad);

				for (int i = 0; i < _vertexList.Length; ++i)
				{
					_sharedNormalIndices[i] = new List<VertexEntry>();
				}
			}

			return true;
		}

		public static int TransferRegularPointAttributesToVertices(int[] vertexList, ref HAPI_AttributeInfo attribInfo, float[] inData, ref float[] outData)
		{
			int validWedgeCount = 0;
			if (attribInfo.exists && attribInfo.tupleSize > 0)
			{
				int wedgeCount = vertexList.Length;

				// Re-indexed wedges
				outData = new float[wedgeCount * attribInfo.tupleSize];

				int lastValidWedgeIndex = 0;
				for (int wedgeIndex = 0; wedgeIndex < wedgeCount; ++wedgeIndex)
				{
					int vertexIndex = vertexList[wedgeIndex];
					if (vertexIndex == -1)
					{
						continue;
					}

					validWedgeCount++;

					int primIndex = wedgeIndex / 3;
					int saveIndex = 0;
					float value = 0;

					for (int attribIndex = 0; attribIndex < attribInfo.tupleSize; ++attribIndex)
					{
						switch (attribInfo.owner)
						{
							case HAPI_AttributeOwner.HAPI_ATTROWNER_POINT:
							{
								value = inData[vertexIndex * attribInfo.tupleSize + attribIndex];
								break;
							}
							case HAPI_AttributeOwner.HAPI_ATTROWNER_PRIM:
							{
								value = inData[primIndex * attribInfo.tupleSize + attribIndex];
								break;
							}
							case HAPI_AttributeOwner.HAPI_ATTROWNER_DETAIL:
							{
								value = inData[attribIndex];
								break;
							}
							case HAPI_AttributeOwner.HAPI_ATTROWNER_VERTEX:
							{
								value = inData[wedgeIndex * attribInfo.tupleSize + attribIndex];
								break;
							}
							default:
							{
								Debug.LogAssertion("Unsupported attribute owner " + attribInfo.owner);
								continue;
							}
						}

						saveIndex = lastValidWedgeIndex * attribInfo.tupleSize + attribIndex;
						outData[saveIndex] = value;
					}

					lastValidWedgeIndex++;
				}
			}

			return validWedgeCount;
		}
	}

	/// <summary>
	/// Helper used for storing vertex connections for normal generation
	/// </summary>
	public class VertexEntry
	{
		public int _meshKey;
		public int _vertexIndex;
		public int _normalIndex;

		public VertexEntry(int meshKey, int vertexIndex, int normalIndex)
		{
			_meshKey = meshKey;
			_vertexIndex = vertexIndex;
			_normalIndex = normalIndex;
		}
	}

}   // HoudiniEngineUnity