﻿/*
* Copyright (c) <2017> Side Effects Software Inc.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
* Produced by:
*      Side Effects Software Inc
*      123 Front Street West, Suite 1401
*      Toronto, Ontario
*      Canada   M5J 2M2
*      416-504-9876
*
*/

using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace HoudiniEngineUnity
{
	/// <summary>
	/// Generic serializable Dictionary.
	/// </summary>
	/// <typeparam name="TKey"></typeparam>
	/// <typeparam name="TValue"></typeparam>
	[System.Serializable]
	public class HEU_SerializableDictionary<TKey, TValue> : IDictionary<TKey, TValue>, UnityEngine.ISerializationCallbackReceiver
	{
		[System.NonSerialized]
		private Dictionary<TKey, TValue> _dictionary;

		[SerializeField]
		private TKey[] _keys;

		[SerializeField]
		private TValue[] _values;


		public TValue this[TKey key]
		{
			get
			{
				if(_dictionary == null)
				{
					throw new KeyNotFoundException();
				}
				return _dictionary[key];
			}
			set
			{
				if(_dictionary == null)
				{
					_dictionary = new Dictionary<TKey, TValue>();
				}
				_dictionary[key] = value;
			}
		}

		public ICollection<TKey> Keys
		{
			get
			{
				if(_dictionary == null)
				{
					_dictionary = new Dictionary<TKey, TValue>();
				}
				return _dictionary.Keys;
			}
		}

		public ICollection<TValue> Values
		{
			get
			{
				if(_dictionary == null)
				{
					_dictionary = new Dictionary<TKey, TValue>();
				}
				return _dictionary.Values;
			}
		}

		public int Count
		{
			get { return (_dictionary != null) ? _dictionary.Count : 0; }
		}

		public bool IsReadOnly
		{
			get { return false; }
		}

		public void Add(TKey key, TValue value)
		{
			if (_dictionary == null)
			{
				_dictionary = new Dictionary<TKey, TValue>();
			}
			_dictionary.Add(key, value);
		}

		public void Add(KeyValuePair<TKey, TValue> item)
		{
			if (_dictionary == null)
			{
				_dictionary = new Dictionary<TKey, TValue>();
			}
			(_dictionary as ICollection<KeyValuePair<TKey, TValue>>).Add(item);
		}

		public void Clear()
		{
			if(_dictionary != null)
			{
				_dictionary.Clear();
			}
		}

		public bool Contains(KeyValuePair<TKey, TValue> item)
		{
			if (_dictionary == null)
			{
				return false;
			}
			return (_dictionary as ICollection<KeyValuePair<TKey, TValue>>).Contains(item);
		}

		public bool ContainsKey(TKey key)
		{
			if (_dictionary == null)
			{
				return false;
			}
			return _dictionary.ContainsKey(key);
		}

		public void CopyTo(KeyValuePair<TKey, TValue>[] array, int arrayIndex)
		{
			if (_dictionary == null)
			{
				return;
			}
			(_dictionary as ICollection<KeyValuePair<TKey, TValue>>).CopyTo(array, arrayIndex);
		}

		public IEnumerator<KeyValuePair<TKey, TValue>> GetEnumerator()
		{
			if (_dictionary == null)
			{
				return default(Dictionary<TKey, TValue>.Enumerator);
			}
			return _dictionary.GetEnumerator();
		}

		public bool Remove(TKey key)
		{
			if (_dictionary == null)
			{
				return false;
			}
			return _dictionary.Remove(key);
		}

		public bool Remove(KeyValuePair<TKey, TValue> item)
		{
			if (_dictionary == null)
			{
				return false;
			}
			return (_dictionary as ICollection<KeyValuePair<TKey, TValue>>).Remove(item);
		}

		public bool TryGetValue(TKey key, out TValue value)
		{
			if(_dictionary == null)
			{
				value = default(TValue);
				return false;
			}
			return _dictionary.TryGetValue(key, out value);
		}

		IEnumerator IEnumerable.GetEnumerator()
		{
			if(_dictionary == null)
			{
				_dictionary = new Dictionary<TKey, TValue>();
			}
			return _dictionary.GetEnumerator();
		}

		public void OnAfterDeserialize()
		{
			if (_keys != null && _values != null)
			{
				// Read keys and values array into dictionary
				if(_dictionary == null)
				{
					_dictionary = new Dictionary<TKey, TValue>(_keys.Length);
				}
				else
				{
					_dictionary.Clear();
				}

				for(int i = 0; i < _keys.Length; ++i)
				{
					if(i < _values.Length)
					{
						_dictionary[_keys[i]] = _values[i];
					}
					else
					{
						_dictionary[_keys[i]] = default(TValue);
					}
				}
			}

			_keys = null;
			_values = null;
		}

		public void OnBeforeSerialize()
		{
			if (_dictionary == null || _dictionary.Count == 0)
			{
				_keys = null;
				_values = null;
			}
			else
			{
				// Copy dictionary into keys and values array
				int itemCount = _dictionary.Count;
				_keys = new TKey[itemCount];
				_values = new TValue[itemCount];

				int index = 0;
				var enumerator = _dictionary.GetEnumerator();
				while(enumerator.MoveNext())
				{
					_keys[index] = enumerator.Current.Key;
					_values[index] = enumerator.Current.Value;
					index++;
				}
			}
		}
	}

}   // HoudiniEngineUnity